<?php
session_start();
require __DIR__ . '/../config/db.php';

// Si ya está logueado, redirigir al dashboard
if (isset($_SESSION['usuario'])) {
    header("Location: /inventario/index.php");
    exit;
}

$error = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $usuario = trim($_POST['usuario']);
    $clave   = $_POST['clave'];
    
    // Validaciones básicas
    if (empty($usuario) || empty($clave)) {
        $error = "Usuario y contraseña son obligatorios.";
    } else {
        try {
            // Buscar usuario activo
            $stmt = $pdo->prepare("SELECT * FROM usuarios WHERE usuario = ? AND activo = 1 LIMIT 1");
            $stmt->execute([$usuario]);
            $user = $stmt->fetch();

            if ($user && password_verify($clave, $user['clave'])) {
                // Contraseña correcta, crear sesión
                $_SESSION['usuario'] = [
                    'id' => $user['id'],
                    'usuario' => $user['usuario'],
                    'rol' => $user['rol'],
                    'nombre' => $user['nombre'] ?? $user['usuario']
                ];
                
                // Actualizar último login
                $stmt = $pdo->prepare("UPDATE usuarios SET ultimo_login = NOW(), actualizado_en = NOW() WHERE id = ?");
                $stmt->execute([$user['id']]);
                
                // Registrar acceso en auditoría si la tabla existe
                try {
                    $stmt = $pdo->prepare("
                        INSERT INTO auditoria (tabla, accion, registro_id, valores_nuevos, usuario_id) 
                        VALUES ('usuarios', 'LOGIN', ?, ?, ?)
                    ");
                    $stmt->execute([
                        $user['id'], 
                        json_encode(['ip' => $_SERVER['REMOTE_ADDR'], 'user_agent' => $_SERVER['HTTP_USER_AGENT']]),
                        $user['id']
                    ]);
                } catch (Exception $e) {
                    // Si falla la auditoría, continuar igual
                    error_log("Error en auditoría de login: " . $e->getMessage());
                }
                
                // Redirigir a la URL guardada o al dashboard
                $redirect_url = $_SESSION['redirect_url'] ?? '/inventario/index.php';
                unset($_SESSION['redirect_url']);
                header("Location: $redirect_url");
                exit;
                
            } else {
                // Credenciales incorrectas
                $error = "Usuario o contraseña incorrectos.";
                
                // Registrar intento fallido
                error_log("Intento de login fallido para usuario: " . $usuario . " desde IP: " . $_SERVER['REMOTE_ADDR']);
            }
            
        } catch (Exception $e) {
            $error = "Error del sistema. Por favor intenta más tarde.";
            error_log("Error en login: " . $e->getMessage());
        }
    }
}

// Mensajes de error por parámetros GET
if (isset($_GET['timeout'])) {
    $error = "Sesión expirada por inactividad. Por favor ingresa nuevamente.";
} elseif (isset($_GET['invalid'])) {
    $error = "Usuario no válido o inactivo. Contacta al administrador.";
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Login - Inventario Jusocar</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.1/font/bootstrap-icons.css">
    <style>
        body {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        .login-card {
            background: white;
            border-radius: 15px;
            box-shadow: 0 15px 35px rgba(0,0,0,0.1);
            overflow: hidden;
            max-width: 400px;
            width: 100%;
        }
        .login-header {
            background: linear-gradient(135deg, #007bff, #0056b3);
            color: white;
            padding: 2rem;
            text-align: center;
        }
        .login-body {
            padding: 2rem;
        }
        .form-control:focus {
            border-color: #007bff;
            box-shadow: 0 0 0 0.2rem rgba(0,123,255,.25);
        }
        .btn-login {
            background: linear-gradient(135deg, #007bff, #0056b3);
            border: none;
            padding: 12px;
            font-weight: 600;
        }
        .btn-login:hover {
            background: linear-gradient(135deg, #0056b3, #004494);
            transform: translateY(-1px);
            box-shadow: 0 5px 15px rgba(0,0,0,0.2);
        }
    </style>
</head>
<body>
<div class="login-card">
    <div class="login-header">
        <h2><i class="bi bi-shield-lock"></i></h2>
        <h4 class="mb-0">Control Inventario</h4>
        <p class="mb-0 opacity-75">Jusocar</p>
    </div>
    <div class="login-body">
        <?php if($error): ?>
            <div class="alert alert-danger alert-dismissible fade show">
                <i class="bi bi-exclamation-triangle"></i> <?= $error ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        <?php endif; ?>

        <form method="POST">
            <div class="mb-3">
                <label class="form-label">Usuario</label>
                <div class="input-group">
                    <span class="input-group-text"><i class="bi bi-person"></i></span>
                    <input type="text" name="usuario" class="form-control" placeholder="Ingresa tu usuario" required autofocus>
                </div>
            </div>
            
            <div class="mb-4">
                <label class="form-label">Contraseña</label>
                <div class="input-group">
                    <span class="input-group-text"><i class="bi bi-key"></i></span>
                    <input type="password" name="clave" class="form-control" placeholder="Ingresa tu contraseña" required>
                </div>
            </div>
            
            <button type="submit" class="btn btn-login btn-primary w-100 mb-3">
                <i class="bi bi-box-arrow-in-right"></i> Ingresar al Sistema
            </button>
            
            <div class="text-center">
                <small class="text-muted">
                    Sistema de Control de Inventario v2.0
                </small>
            </div>
        </form>
    </div>
    <div class="card-footer text-center text-muted py-3">
        <small>&copy; <?= date('Y') ?> Jusocar - Todos los derechos reservados</small>
    </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
<script>
// Efectos de interacción
document.addEventListener('DOMContentLoaded', function() {
    const inputs = document.querySelectorAll('input');
    inputs.forEach(input => {
        input.addEventListener('focus', function() {
            this.parentElement.classList.add('focus');
        });
        input.addEventListener('blur', function() {
            this.parentElement.classList.remove('focus');
        });
    });
});
</script>
</body>
</html>