<?php
require __DIR__ . '/../includes/auth_check.php';
require __DIR__ . '/../config/db.php';
include __DIR__ . '/../includes/header.php';

// Solo administradores pueden registrar usuarios
if (!tienePermiso('admin')) {
    header("Location: /inventario/index.php");
    exit;
}

$mensaje = '';
$error = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $usuario = trim($_POST['usuario']);
    $nombre = trim($_POST['nombre']);
    $email = trim($_POST['email']);
    $clave = $_POST['password'];
    $confirmar_clave = $_POST['confirm_password'];
    $rol = $_POST['rol'];

    // Validaciones
    $errores = [];

    if (empty($usuario) || strlen($usuario) < 3) {
        $errores[] = "El usuario debe tener al menos 3 caracteres.";
    }

    if (empty($nombre)) {
        $errores[] = "El nombre es obligatorio.";
    }

    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $errores[] = "El email no es válido.";
    }

    if (strlen($clave) < 6) {
        $errores[] = "La contraseña debe tener al menos 6 caracteres.";
    }

    if ($clave !== $confirmar_clave) {
        $errores[] = "Las contraseñas no coinciden.";
    }

    if (empty($rol) || !in_array($rol, ['admin', 'operador', 'auditor'])) {
        $errores[] = "Rol no válido.";
    }

    // Verificar si el usuario o email ya existen
    if (empty($errores)) {
        try {
            $stmt = $pdo->prepare("SELECT id FROM usuarios WHERE usuario = ? OR email = ?");
            $stmt->execute([$usuario, $email]);
            if ($stmt->fetch()) {
                $errores[] = "El usuario o email ya existen.";
            }
        } catch (Exception $e) {
            $errores[] = "Error verificando usuario existente.";
        }
    }

    if (empty($errores)) {
        try {
            $pdo->beginTransaction();

            $hash_clave = password_hash($clave, PASSWORD_DEFAULT);
            
            $stmt = $pdo->prepare("INSERT INTO usuarios (usuario, nombre, email, clave, rol) VALUES (?, ?, ?, ?, ?)");
            $stmt->execute([$usuario, $nombre, $email, $hash_clave, $rol]);

            $usuario_id = $pdo->lastInsertId();

            // Registrar en auditoría
            try {
                $stmt = $pdo->prepare("
                    INSERT INTO auditoria (tabla, accion, registro_id, valores_nuevos, usuario_id) 
                    VALUES ('usuarios', 'INSERT', ?, ?, ?)
                ");
                $stmt->execute([
                    $usuario_id,
                    json_encode([
                        'usuario' => $usuario,
                        'nombre' => $nombre,
                        'email' => $email,
                        'rol' => $rol
                    ]),
                    $_SESSION['usuario']['id']
                ]);
            } catch (Exception $e) {
                // Continuar si falla la auditoría
                error_log("Error en auditoría de registro: " . $e->getMessage());
            }

            $pdo->commit();

            $mensaje = "✅ Usuario registrado correctamente.";
            
            // Limpiar formulario
            $_POST = [];

        } catch (Exception $e) {
            $pdo->rollBack();
            $error = "❌ Error al registrar usuario: " . $e->getMessage();
        }
    } else {
        $error = "❌ Errores encontrados:<br>" . implode("<br>", $errores);
    }
}

// Obtener lista de usuarios existentes
$usuarios = $pdo->query("
    SELECT id, usuario, nombre, email, rol, activo, ultimo_login, creado_en 
    FROM usuarios 
    ORDER BY creado_en DESC
")->fetchAll();
?>
<h3 class="mb-4">👤 Registrar Nuevo Usuario</h3>

<?php if($mensaje): ?>
    <div class="alert alert-success"><?= $mensaje ?></div>
<?php endif; ?>

<?php if($error): ?>
    <div class="alert alert-danger"><?= $error ?></div>
<?php endif; ?>

<div class="row">
    <div class="col-md-6">
        <form method="POST" class="card p-4 shadow-sm">
            <div class="mb-3">
                <label class="form-label">Usuario <span class="text-danger">*</span></label>
                <input type="text" name="usuario" value="<?= isset($_POST['usuario']) ? htmlspecialchars($_POST['usuario']) : '' ?>" 
                       class="form-control" required minlength="3" maxlength="50"
                       placeholder="Nombre de usuario para login">
            </div>

            <div class="mb-3">
                <label class="form-label">Nombre Completo <span class="text-danger">*</span></label>
                <input type="text" name="nombre" value="<?= isset($_POST['nombre']) ? htmlspecialchars($_POST['nombre']) : '' ?>" 
                       class="form-control" required maxlength="100"
                       placeholder="Nombre real del usuario">
            </div>

            <div class="mb-3">
                <label class="form-label">Email <span class="text-danger">*</span></label>
                <input type="email" name="email" value="<?= isset($_POST['email']) ? htmlspecialchars($_POST['email']) : '' ?>" 
                       class="form-control" required maxlength="100"
                       placeholder="email@ejemplo.com">
            </div>

            <div class="row">
                <div class="col-md-6">
                    <div class="mb-3">
                        <label class="form-label">Contraseña <span class="text-danger">*</span></label>
                        <input type="password" name="password" class="form-control" required minlength="6"
                               placeholder="Mínimo 6 caracteres">
                    </div>
                </div>
                <div class="col-md-6">
                    <div class="mb-3">
                        <label class="form-label">Confirmar Contraseña <span class="text-danger">*</span></label>
                        <input type="password" name="confirm_password" class="form-control" required minlength="6"
                               placeholder="Repite la contraseña">
                    </div>
                </div>
            </div>

            <div class="mb-3">
                <label class="form-label">Rol <span class="text-danger">*</span></label>
                <select name="rol" class="form-select" required>
                    <option value="">-- Seleccionar Rol --</option>
                    <option value="admin" <?= (isset($_POST['rol']) && $_POST['rol'] == 'admin') ? 'selected' : '' ?>>Administrador</option>
                    <option value="operador" <?= (isset($_POST['rol']) && $_POST['rol'] == 'operador') ? 'selected' : '' ?>>Operador</option>
                    <option value="auditor" <?= (isset($_POST['rol']) && $_POST['rol'] == 'auditor') ? 'selected' : '' ?>>Auditor</option>
                </select>
                <div class="form-text">
                    <strong>Administrador:</strong> Acceso completo<br>
                    <strong>Operador:</strong> Puede gestionar inventario pero no usuarios<br>
                    <strong>Auditor:</strong> Solo puede ver reportes e historial
                </div>
            </div>

            <button type="submit" class="btn btn-primary w-100 py-2">
                <i class="bi bi-person-plus"></i> Registrar Usuario
            </button>
        </form>
    </div>

    <div class="col-md-6">
        <div class="card">
            <div class="card-header bg-light">
                <h6 class="mb-0">👥 Usuarios Existentes</h6>
            </div>
            <div class="card-body">
                <div class="table-responsive">
                    <table class="table table-sm">
                        <thead>
                            <tr>
                                <th>Usuario</th>
                                <th>Nombre</th>
                                <th>Rol</th>
                                <th>Estado</th>
                                <th>Último Login</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($usuarios as $user): ?>
                                <tr>
                                    <td>
                                        <strong><?= htmlspecialchars($user['usuario']) ?></strong>
                                        <?php if ($user['id'] == $_SESSION['usuario']['id']): ?>
                                            <span class="badge bg-info">Tú</span>
                                        <?php endif; ?>
                                    </td>
                                    <td><?= htmlspecialchars($user['nombre']) ?></td>
                                    <td>
                                        <span class="badge bg-<?= $user['rol'] == 'admin' ? 'danger' : ($user['rol'] == 'operador' ? 'warning' : 'secondary') ?>">
                                            <?= $user['rol'] ?>
                                        </span>
                                    </td>
                                    <td>
                                        <span class="badge bg-<?= $user['activo'] ? 'success' : 'danger' ?>">
                                            <?= $user['activo'] ? 'Activo' : 'Inactivo' ?>
                                        </span>
                                    </td>
                                    <td>
                                        <small>
                                            <?= $user['ultimo_login'] ? date('d/m/Y H:i', strtotime($user['ultimo_login'])) : 'Nunca' ?>
                                        </small>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
                <div class="text-center text-muted mt-2">
                    <small>Total: <?= count($usuarios) ?> usuarios</small>
                </div>
            </div>
        </div>
    </div>
</div>

<?php include __DIR__ . '/../includes/footer.php'; ?>