<?php
require __DIR__ . '/../includes/auth_check.php';
require __DIR__ . '/../config/db.php';
include __DIR__ . '/../includes/header.php';

if ($_SESSION['usuario']['rol'] !== 'admin') {
    die("Acceso denegado.");
}

$mensaje = '';
$error = '';
$categoria_editando = null;

// Modo edición
$editar_id = $_GET['editar'] ?? null;
if ($editar_id) {
    $stmt = $pdo->prepare("SELECT * FROM categorias WHERE id = ?");
    $stmt->execute([$editar_id]);
    $categoria_editando = $stmt->fetch();
    
    if (!$categoria_editando) {
        $error = "❌ Categoría no encontrada.";
        $editar_id = null;
    }
}

// Procesar formulario (crear o actualizar)
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $nombre = trim($_POST['nombre']);
    $descripcion = trim($_POST['descripcion']);
    $categoria_id = $_POST['categoria_id'] ?? null;

    // Validaciones
    $errores = [];

    if (empty($nombre)) {
        $errores[] = "El nombre de la categoría es obligatorio.";
    }

    if (strlen($nombre) < 2) {
        $errores[] = "El nombre debe tener al menos 2 caracteres.";
    }

    // Verificar si el nombre ya existe (excluyendo la categoría actual en edición)
    if (empty($errores)) {
        $sql = "SELECT id FROM categorias WHERE nombre = ?";
        $params = [$nombre];
        
        if ($categoria_id) {
            $sql .= " AND id != ?";
            $params[] = $categoria_id;
        }
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        
        if ($stmt->fetch()) {
            $errores[] = "Ya existe una categoría con ese nombre.";
        }
    }

    if (empty($errores)) {
        try {
            $pdo->beginTransaction();

            if ($categoria_id) {
                // MODE: Actualizar categoría existente
                $stmt = $pdo->prepare("UPDATE categorias SET nombre = ?, descripcion = ?, actualizado_en = CURRENT_TIMESTAMP WHERE id = ?");
                $stmt->execute([$nombre, $descripcion, $categoria_id]);
                
                $mensaje = "✅ Categoría actualizada correctamente.";
                
                // Registrar en auditoría
                try {
                    $stmt = $pdo->prepare("
                        INSERT INTO auditoria (tabla, accion, registro_id, valores_nuevos, usuario_id) 
                        VALUES ('categorias', 'UPDATE', ?, ?, ?)
                    ");
                    $stmt->execute([
                        $categoria_id,
                        json_encode(['nombre' => $nombre, 'descripcion' => $descripcion]),
                        $_SESSION['usuario']['id']
                    ]);
                } catch (Exception $e) {
                    error_log("Error en auditoría de categoría: " . $e->getMessage());
                }
                
            } else {
                // MODE: Crear nueva categoría
                $stmt = $pdo->prepare("INSERT INTO categorias (nombre, descripcion) VALUES (?, ?)");
                $stmt->execute([$nombre, $descripcion]);
                $categoria_id = $pdo->lastInsertId();
                
                $mensaje = "✅ Categoría creada correctamente.";
                
                // Registrar en auditoría
                try {
                    $stmt = $pdo->prepare("
                        INSERT INTO auditoria (tabla, accion, registro_id, valores_nuevos, usuario_id) 
                        VALUES ('categorias', 'INSERT', ?, ?, ?)
                    ");
                    $stmt->execute([
                        $categoria_id,
                        json_encode(['nombre' => $nombre, 'descripcion' => $descripcion]),
                        $_SESSION['usuario']['id']
                    ]);
                } catch (Exception $e) {
                    error_log("Error en auditoría de categoría: " . $e->getMessage());
                }
            }

            $pdo->commit();

            // Limpiar formulario después de éxito (solo si no estamos editando)
            if (!$categoria_id) {
                $_POST = [];
            } else {
                // Recargar datos de la categoría editada
                $stmt = $pdo->prepare("SELECT * FROM categorias WHERE id = ?");
                $stmt->execute([$categoria_id]);
                $categoria_editando = $stmt->fetch();
            }

        } catch (Exception $e) {
            $pdo->rollBack();
            $error = "❌ Error al " . ($categoria_id ? "actualizar" : "crear") . " categoría: " . $e->getMessage();
        }
    } else {
        $error = "❌ Errores encontrados:<br>" . implode("<br>", $errores);
    }
}

// Eliminar categoría
if (isset($_GET['eliminar'])) {
    $eliminar_id = $_GET['eliminar'];
    
    // Verificar si la categoría tiene productos asociados
    $stmt = $pdo->prepare("SELECT COUNT(*) as total FROM cintas WHERE categoria_id = ? AND activo = 1");
    $stmt->execute([$eliminar_id]);
    $productos_asociados = $stmt->fetchColumn();
    
    if ($productos_asociados > 0) {
        $error = "❌ No se puede eliminar la categoría porque tiene $productos_asociados producto(s) asociado(s).";
    } else {
        try {
            $pdo->beginTransaction();
            
            // Obtener datos antes de eliminar para auditoría
            $stmt = $pdo->prepare("SELECT * FROM categorias WHERE id = ?");
            $stmt->execute([$eliminar_id]);
            $categoria_eliminada = $stmt->fetch();
            
            $stmt = $pdo->prepare("DELETE FROM categorias WHERE id = ?");
            $stmt->execute([$eliminar_id]);
            
            // Registrar en auditoría
            try {
                $stmt = $pdo->prepare("
                    INSERT INTO auditoria (tabla, accion, registro_id, valores_anteriores, usuario_id) 
                    VALUES ('categorias', 'DELETE', ?, ?, ?)
                ");
                $stmt->execute([
                    $eliminar_id,
                    json_encode($categoria_eliminada),
                    $_SESSION['usuario']['id']
                ]);
            } catch (Exception $e) {
                error_log("Error en auditoría de categoría: " . $e->getMessage());
            }
            
            $pdo->commit();
            $mensaje = "✅ Categoría eliminada correctamente.";
            
        } catch (Exception $e) {
            $pdo->rollBack();
            $error = "❌ Error al eliminar categoría: " . $e->getMessage();
        }
    }
}

// Obtener todas las categorías con estadísticas
$categorias = $pdo->query("
    SELECT c.*, 
           COUNT(prod.id) as total_productos,
           SUM(CASE WHEN prod.stock < prod.stock_minimo THEN 1 ELSE 0 END) as productos_stock_bajo,
           SUM(CASE WHEN prod.stock = 0 THEN 1 ELSE 0 END) as productos_sin_stock
    FROM categorias c
    LEFT JOIN cintas prod ON c.id = prod.categoria_id AND prod.activo = 1
    GROUP BY c.id, c.nombre, c.descripcion, c.creado_en, c.actualizado_en
    ORDER BY c.nombre ASC
")->fetchAll();

// Estadísticas generales
$stats = $pdo->query("
    SELECT 
        COUNT(*) as total_categorias,
        SUM(CASE WHEN total_productos > 0 THEN 1 ELSE 0 END) as categorias_con_productos
    FROM (
        SELECT c.id, COUNT(prod.id) as total_productos
        FROM categorias c
        LEFT JOIN cintas prod ON c.id = prod.categoria_id AND prod.activo = 1
        GROUP BY c.id
    ) as cat_stats
")->fetch();
?>
<div class="row">
    <div class="col-lg-6">
        <h3 class="mb-4">
            <?= $editar_id ? '✏️ Editar Categoría' : '📂 Agregar Categoría' ?>
        </h3>

        <?php if($mensaje): ?>
            <div class="alert alert-success alert-dismissible fade show">
                <?= $mensaje ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        <?php endif; ?>

        <?php if($error): ?>
            <div class="alert alert-danger alert-dismissible fade show">
                <?= $error ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        <?php endif; ?>

        <div class="card border-0 shadow-sm">
            <div class="card-header bg-gradient-primary text-white py-3">
                <h5 class="mb-0">
                    <?= $editar_id ? 'Editar Categoría' : 'Nueva Categoría' ?>
                </h5>
            </div>
            <div class="card-body p-4">
                <form method="POST">
                    <?php if ($editar_id): ?>
                        <input type="hidden" name="categoria_id" value="<?= $editar_id ?>">
                    <?php endif; ?>
                    
                    <div class="mb-3">
                        <label class="form-label fw-semibold">Nombre de la Categoría <span class="text-danger">*</span></label>
                        <input type="text" name="nombre" 
                               value="<?= $categoria_editando ? htmlspecialchars($categoria_editando['nombre']) : (isset($_POST['nombre']) ? htmlspecialchars($_POST['nombre']) : '') ?>" 
                               class="form-control rounded-pill" required minlength="2" maxlength="100"
                               placeholder="Ej: Sarga, Trevira, Cinta colchón...">
                        <div class="form-text">Nombre único para identificar la categoría (mínimo 2 caracteres).</div>
                    </div>

                    <div class="mb-4">
                        <label class="form-label fw-semibold">Descripción</label>
                        <textarea name="descripcion" class="form-control" rows="4" 
                                  placeholder="Descripción opcional de la categoría..."><?= $categoria_editando ? htmlspecialchars($categoria_editando['descripcion']) : (isset($_POST['descripcion']) ? htmlspecialchars($_POST['descripcion']) : '') ?></textarea>
                    </div>

                    <div class="d-grid gap-2 d-md-flex justify-content-md-end">
                        <?php if ($editar_id): ?>
                            <a href="agregar.php" class="btn btn-outline-secondary rounded-pill me-md-2">
                                <i class="bi bi-x-circle me-1"></i>Cancelar
                            </a>
                        <?php endif; ?>
                        <button type="submit" class="btn btn-primary rounded-pill px-4">
                            <i class="bi bi-<?= $editar_id ? 'check' : 'plus' ?>-circle me-1"></i>
                            <?= $editar_id ? 'Actualizar Categoría' : 'Crear Categoría' ?>
                        </button>
                    </div>
                </form>
            </div>
        </div>
        
        <!-- Información de Estadísticas -->
        <div class="card border-0 shadow-sm mt-4">
            <div class="card-header bg-gradient-info text-white py-3">
                <h6 class="mb-0">📊 Estadísticas de Categorías</h6>
            </div>
            <div class="card-body">
                <div class="row text-center">
                    <div class="col-6">
                        <div class="border rounded p-3 hover-lift">
                            <div class="h4 mb-1 text-primary fw-bold"><?= $stats['total_categorias'] ?></div>
                            <small class="text-muted">Total Categorías</small>
                        </div>
                    </div>
                    <div class="col-6">
                        <div class="border rounded p-3 hover-lift">
                            <div class="h4 mb-1 text-success fw-bold"><?= $stats['categorias_con_productos'] ?></div>
                            <small class="text-muted">Con Productos</small>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <div class="col-lg-6">
        <h3 class="mb-4">📋 Categorías Existentes</h3>
        
        <?php if (empty($categorias)): ?>
            <div class="card border-0 shadow-sm">
                <div class="card-body text-center py-5">
                    <i class="bi bi-tags display-1 text-muted opacity-25"></i>
                    <h5 class="mt-3 text-muted">No hay categorías creadas</h5>
                    <p class="text-muted">Comienza creando tu primera categoría</p>
                </div>
            </div>
        <?php else: ?>
            <div class="card border-0 shadow-sm">
                <div class="card-header bg-gradient-light py-3">
                    <h6 class="mb-0">Lista de Categorías (<?= count($categorias) ?>)</h6>
                </div>
                <div class="card-body p-0">
                    <div class="list-group list-group-flush">
                        <?php foreach ($categorias as $cat): ?>
                            <div class="list-group-item p-4 border-0 <?= $editar_id == $cat['id'] ? 'bg-warning bg-opacity-10' : '' ?>">
                                <div class="d-flex justify-content-between align-items-start">
                                    <div class="flex-grow-1">
                                        <div class="d-flex align-items-center mb-2">
                                            <h6 class="mb-0 fw-bold text-primary"><?= htmlspecialchars($cat['nombre']) ?></h6>
                                            <?php if ($editar_id == $cat['id']): ?>
                                                <span class="badge bg-warning text-dark rounded-pill ms-2">Editando</span>
                                            <?php endif; ?>
                                        </div>
                                        
                                        <?php if (!empty($cat['descripcion'])): ?>
                                            <p class="text-muted small mb-2"><?= htmlspecialchars($cat['descripcion']) ?></p>
                                        <?php endif; ?>
                                        
                                        <div class="d-flex gap-3 small text-muted">
                                            <span>
                                                <i class="bi bi-box-seam me-1"></i>
                                                <?= $cat['total_productos'] ?> producto(s)
                                            </span>
                                            <?php if ($cat['productos_stock_bajo'] > 0): ?>
                                                <span class="text-warning">
                                                    <i class="bi bi-exclamation-triangle me-1"></i>
                                                    <?= $cat['productos_stock_bajo'] ?> bajo stock
                                                </span>
                                            <?php endif; ?>
                                            <?php if ($cat['productos_sin_stock'] > 0): ?>
                                                <span class="text-danger">
                                                    <i class="bi bi-x-circle me-1"></i>
                                                    <?= $cat['productos_sin_stock'] ?> sin stock
                                                </span>
                                            <?php endif; ?>
                                        </div>
                                        
                                        <small class="text-muted">
                                            <i class="bi bi-calendar me-1"></i>
                                            Creada: <?= date('d/m/Y', strtotime($cat['creado_en'])) ?>
                                            <?php if ($cat['creado_en'] != $cat['actualizado_en']): ?>
                                                | Actualizada: <?= date('d/m/Y', strtotime($cat['actualizado_en'])) ?>
                                            <?php endif; ?>
                                        </small>
                                    </div>
                                    
                                    <div class="btn-group btn-group-sm ms-3">
                                        <a href="agregar.php?editar=<?= $cat['id'] ?>" 
                                           class="btn btn-outline-warning rounded-pill px-3" 
                                           title="Editar categoría">
                                            <i class="bi bi-pencil"></i>
                                        </a>
                                        <?php if ($cat['total_productos'] == 0): ?>
                                            <a href="agregar.php?eliminar=<?= $cat['id'] ?>" 
                                               class="btn btn-outline-danger rounded-pill px-3" 
                                               title="Eliminar categoría"
                                               onclick="return confirm('¿Estás seguro de que quieres eliminar la categoría \"<?= htmlspecialchars($cat['nombre']) ?>\"?')">
                                                <i class="bi bi-trash"></i>
                                            </a>
                                        <?php else: ?>
                                            <span class="btn btn-outline-secondary rounded-pill px-3" 
                                                  title="No se puede eliminar (tiene productos asociados)">
                                                <i class="bi bi-trash"></i>
                                            </span>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </div>
            </div>
            
            <!-- Información de uso -->
            <div class="alert alert-info mt-4 border-0 shadow-sm">
                <div class="d-flex">
                    <div class="flex-shrink-0">
                        <i class="bi bi-info-circle fs-4"></i>
                    </div>
                    <div class="flex-grow-1 ms-3">
                        <h6 class="alert-heading">💡 Tip de uso</h6>
                        <p class="mb-0 small">
                            Las categorías solo se pueden eliminar si no tienen productos asociados. 
                            Para reorganizar productos entre categorías, edita cada producto individualmente.
                        </p>
                    </div>
                </div>
            </div>
        <?php endif; ?>
    </div>
</div>

<style>
.hover-lift:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 8px rgba(0,0,0,0.1);
    transition: all 0.3s ease;
}

.list-group-item:hover {
    background-color: #f8f9fa;
    transition: background-color 0.2s ease;
}

.bg-gradient-primary {
    background: linear-gradient(135deg, #007bff, #0056b3);
}

.bg-gradient-light {
    background: linear-gradient(135deg, #f8f9fa, #e9ecef);
}

.bg-gradient-info {
    background: linear-gradient(135deg, #17a2b8, #138496);
}

.rounded-pill {
    border-radius: 50rem !important;
}

.border-0 {
    border: 0 !important;
}

.shadow-sm {
    box-shadow: 0 0.125rem 0.25rem rgba(0, 0, 0, 0.075) !important;
}
</style>

<?php include __DIR__ . '/../includes/footer.php'; ?>