<?php
require __DIR__ . '/../includes/auth_check.php';
require __DIR__ . '/../config/db.php';
include __DIR__ . '/../includes/header.php';

if ($_SESSION['usuario']['rol'] !== 'admin') {
    die("Acceso denegado.");
}

$id = $_GET['id'] ?? null;
if (!$id) {
    die("ID de producto no especificado.");
}

// Obtener datos actuales
$stmt = $pdo->prepare("SELECT * FROM cintas WHERE id = ?");
$stmt->execute([$id]);
$cinta = $stmt->fetch();

if (!$cinta) {
    die("Producto no encontrado.");
}

$mensaje = '';
$errores = [];

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $descripcion = trim($_POST['descripcion']);
    $precio = (int)$_POST['precio_unitario'];
    $unidad = $_POST['unidad'];
    $stock = (int)$_POST['stock'];
    $stock_minimo = (int)$_POST['stock_minimo'];
    $categoria_id = $_POST['categoria_id'];

    // Validaciones
    if (empty($descripcion)) {
        $errores[] = "La descripción es obligatoria.";
    }
    
    if ($precio <= 0) {
        $errores[] = "El precio unitario debe ser mayor a cero.";
    }
    
    if ($stock < 0) {
        $errores[] = "El stock no puede ser negativo.";
    }
    
    if ($stock_minimo < 0) {
        $errores[] = "El stock mínimo no puede ser negativo.";
    }

    if (empty($errores)) {
        try {
            $pdo->beginTransaction();

            // Guardar valores anteriores para auditoría
            $valores_anteriores = [
                'descripcion' => $cinta['descripcion'],
                'precio_unitario' => $cinta['precio_unitario'],
                'unidad' => $cinta['unidad'],
                'stock' => $cinta['stock'],
                'stock_minimo' => $cinta['stock_minimo'],
                'categoria_id' => $cinta['categoria_id']
            ];

            $stmt = $pdo->prepare("UPDATE cintas 
                                   SET descripcion = ?, precio_unitario = ?, unidad = ?, 
                                       stock = ?, stock_minimo = ?, categoria_id = ?,
                                       actualizado_en = CURRENT_TIMESTAMP
                                   WHERE id = ?");
            $stmt->execute([$descripcion, $precio, $unidad, $stock, $stock_minimo, $categoria_id, $id]);

            // Registrar en auditoría
            try {
                $valores_nuevos = [
                    'descripcion' => $descripcion,
                    'precio_unitario' => $precio,
                    'unidad' => $unidad,
                    'stock' => $stock,
                    'stock_minimo' => $stock_minimo,
                    'categoria_id' => $categoria_id
                ];
                
                $stmt = $pdo->prepare("
                    INSERT INTO auditoria (tabla, accion, registro_id, valores_anteriores, valores_nuevos, usuario_id) 
                    VALUES ('cintas', 'UPDATE', ?, ?, ?, ?)
                ");
                $stmt->execute([
                    $id,
                    json_encode($valores_anteriores),
                    json_encode($valores_nuevos),
                    $_SESSION['usuario']['id']
                ]);
            } catch (Exception $e) {
                // Continuar si falla la auditoría
                error_log("Error en auditoría de edición: " . $e->getMessage());
            }

            $pdo->commit();

            $mensaje = "✅ Producto actualizado correctamente.";

            // refrescar datos
            $stmt = $pdo->prepare("SELECT * FROM cintas WHERE id = ?");
            $stmt->execute([$id]);
            $cinta = $stmt->fetch();

        } catch (Exception $e) {
            $pdo->rollBack();
            $errores[] = "❌ Error al actualizar producto: " . $e->getMessage();
        }
    }
}

// Obtener categorías
$categorias = $pdo->query("SELECT * FROM categorias ORDER BY nombre")->fetchAll();
?>
<h3 class="mb-4">✏️ Editar Producto</h3>

<?php if(!empty($errores)): ?>
    <div class="alert alert-danger">
        <h6>❌ Errores encontrados:</h6>
        <ul class="mb-0">
            <?php foreach ($errores as $error): ?>
                <li><?= $error ?></li>
            <?php endforeach; ?>
        </ul>
    </div>
<?php endif; ?>

<?php if($mensaje): ?>
    <div class="alert alert-success"><?= $mensaje ?></div>
<?php endif; ?>

<form method="POST" class="card p-4 shadow-sm">
    <div class="row">
        <div class="col-md-6">
            <div class="mb-3">
                <label class="form-label">Código</label>
                <input type="text" value="<?= htmlspecialchars($cinta['codigo']) ?>" class="form-control" disabled>
                <div class="form-text">El código no se puede modificar.</div>
            </div>
        </div>
        <div class="col-md-6">
            <div class="mb-3">
                <label class="form-label">Categoría <span class="text-danger">*</span></label>
                <select name="categoria_id" class="form-select" required>
                    <option value="">-- Seleccionar Categoría --</option>
                    <?php foreach ($categorias as $cat): ?>
                        <option value="<?= $cat['id'] ?>" <?= $cinta['categoria_id']==$cat['id']?'selected':'' ?>>
                            <?= htmlspecialchars($cat['nombre']) ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
        </div>
    </div>

    <div class="mb-3">
        <label class="form-label">Descripción <span class="text-danger">*</span></label>
        <input type="text" name="descripcion" value="<?= htmlspecialchars($cinta['descripcion']) ?>" 
               class="form-control" required maxlength="255"
               placeholder="Descripción detallada del producto">
    </div>

    <div class="row">
        <div class="col-md-3">
            <div class="mb-3">
                <label class="form-label">Unidad de medida <span class="text-danger">*</span></label>
                <select name="unidad" class="form-select" required>
                    <option value="m" <?= $cinta['unidad']=='m'?'selected':'' ?>>Metros (m)</option>
                    <option value="kg" <?= $cinta['unidad']=='kg'?'selected':'' ?>>Kilos (kg)</option>
                    <option value="u" <?= $cinta['unidad']=='u'?'selected':'' ?>>Unidades (u)</option>
                </select>
            </div>
        </div>
        <div class="col-md-3">
            <div class="mb-3">
                <label class="form-label">Precio Unitario (CLP) <span class="text-danger">*</span></label>
                <input type="number" min="1" name="precio_unitario" 
                       value="<?= (int)$cinta['precio_unitario'] ?>" 
                       class="form-control" required
                       placeholder="Ej: 1200">
            </div>
        </div>
        <div class="col-md-3">
            <div class="mb-3">
                <label class="form-label">Stock Actual <span class="text-danger">*</span></label>
                <input type="number" min="0" name="stock" 
                       value="<?= (int)$cinta['stock'] ?>" 
                       class="form-control" required
                       placeholder="Ej: 100">
            </div>
        </div>
        <div class="col-md-3">
            <div class="mb-3">
                <label class="form-label">Stock Mínimo <span class="text-danger">*</span></label>
                <input type="number" min="0" name="stock_minimo" 
                       value="<?= (int)$cinta['stock_minimo'] ?>" 
                       class="form-control" required
                       placeholder="Ej: 5">
                <div class="form-text">Se alertará cuando el stock esté por debajo de este valor.</div>
            </div>
        </div>
    </div>

    <div class="d-grid gap-2">
        <button type="submit" class="btn btn-primary btn-lg py-2">
            💾 Guardar Cambios
        </button>
        <a href="/inventario/cintas/listar.php" class="btn btn-outline-secondary">
            📦 Volver al Inventario
        </a>
    </div>
</form>

<?php include __DIR__ . '/../includes/footer.php'; ?>