<?php
require_once __DIR__ . '/../includes/auth_check.php';
require_once __DIR__ . '/../config/db.php';
include __DIR__ . '/../includes/header.php';

// Función para traducir unidad
function traducirUnidad($u) {
    switch ($u) {
        case 'm': return 'Metros';
        case 'kg': return 'Kilos';
        case 'u': return 'Unidades';
        default: return $u;
    }
}

// Función para obtener clase de alerta según stock y stock mínimo
function getClaseAlerta($stock, $stock_minimo = 5) {
    if ($stock == 0) {
        return 'table-danger'; // Stock cero - urgente
    } elseif ($stock < $stock_minimo) {
        return 'table-warning'; // Stock bajo
    } elseif ($stock < ($stock_minimo * 2)) {
        return 'table-info'; // Stock medio
    }
    return '';
}

$busqueda = $_GET['q'] ?? '';
$categoria_filtro = $_GET['categoria'] ?? '';
$filtro_stock = $_GET['stock'] ?? '';
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$limit = 25;
$offset = ($page - 1) * $limit;

// Construir consulta con filtros
$whereConditions = [];
$params = [];

if (!empty($busqueda)) {
    $whereConditions[] = "(c.codigo LIKE :busqueda OR c.descripcion LIKE :busqueda)";
    $params[':busqueda'] = "%$busqueda%";
}

if (!empty($categoria_filtro)) {
    $whereConditions[] = "c.categoria_id = :categoria_id";
    $params[':categoria_id'] = $categoria_filtro;
}

// Filtro de stock avanzado
if (!empty($filtro_stock)) {
    switch ($filtro_stock) {
        case 'bajo':
            $whereConditions[] = "c.stock < c.stock_minimo AND c.stock > 0";
            break;
        case 'critico':
            $whereConditions[] = "c.stock < (c.stock_minimo * 0.5) AND c.stock > 0";
            break;
        case 'sin_stock':
            $whereConditions[] = "c.stock = 0";
            break;
        case 'stock_ok':
            $whereConditions[] = "c.stock >= c.stock_minimo";
            break;
    }
}

// Solo productos activos
$whereConditions[] = "c.activo = 1";

$whereClause = '';
if (!empty($whereConditions)) {
    $whereClause = 'WHERE ' . implode(' AND ', $whereConditions);
}

// Consulta para contar total
$sqlCount = "SELECT COUNT(*) as total 
             FROM cintas c 
             LEFT JOIN categorias cat ON c.categoria_id = cat.id
             $whereClause";
$stmtCount = $pdo->prepare($sqlCount);
foreach ($params as $key => $value) {
    $stmtCount->bindValue($key, $value);
}
$stmtCount->execute();
$totalProductos = $stmtCount->fetchColumn();
$totalPaginas = ceil($totalProductos / $limit);

// Consulta principal con paginación
$sql = "SELECT c.*, cat.nombre AS categoria, cat.id AS categoria_id
        FROM cintas c 
        LEFT JOIN categorias cat ON c.categoria_id = cat.id
        $whereClause
        ORDER BY cat.nombre, c.codigo
        LIMIT :limit OFFSET :offset";

$stmt = $pdo->prepare($sql);
foreach ($params as $key => $value) {
    $stmt->bindValue($key, $value);
}
$stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
$stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
$stmt->execute();
$cintas = $stmt->fetchAll();

// Obtener categorías para el filtro
$categorias = $pdo->query("SELECT * FROM categorias ORDER BY nombre")->fetchAll();

// Estadísticas rápidas - PARA LOS ENLACES
$stmtStats = $pdo->query("
    SELECT 
        COUNT(*) as total,
        SUM(CASE WHEN stock = 0 THEN 1 ELSE 0 END) as sin_stock,
        SUM(CASE WHEN stock < stock_minimo THEN 1 ELSE 0 END) as stock_bajo,
        SUM(CASE WHEN stock < (stock_minimo * 0.5) THEN 1 ELSE 0 END) as stock_critico,
        SUM(CASE WHEN stock >= stock_minimo THEN 1 ELSE 0 END) as stock_ok
    FROM cintas
    WHERE activo = 1
");
$stats = $stmtStats->fetch();
?>
<h3 class="mb-4">📦 Inventario / Stock</h3>

<!-- Tarjetas de Estadísticas CLICKABLES Mejoradas -->
<div class="row mb-4">
    <div class="col-xl-3 col-md-6 mb-3">
        <a href="?stock=all" class="text-decoration-none">
            <div class="card metric-card bg-gradient-primary text-white hover-lift">
                <div class="card-body text-center py-4">
                    <div class="metric-icon mb-2">📦</div>
                    <h4 class="metric-value fw-bold"><?= number_format($stats['total'], 0, '', '.') ?></h4>
                    <small class="metric-label">Total Productos</small>
                    <div class="mt-2">
                        <small class="opacity-75">🔍 Ver todos</small>
                    </div>
                </div>
            </div>
        </a>
    </div>
    <div class="col-xl-3 col-md-6 mb-3">
        <a href="?stock=bajo" class="text-decoration-none">
            <div class="card metric-card bg-gradient-warning text-white hover-lift">
                <div class="card-body text-center py-4">
                    <div class="metric-icon mb-2">⚠️</div>
                    <h4 class="metric-value fw-bold"><?= number_format($stats['stock_bajo'], 0, '', '.') ?></h4>
                    <small class="metric-label">Stock Bajo</small>
                    <div class="mt-2">
                        <small class="opacity-75">⚠️ Ver alertas</small>
                    </div>
                </div>
            </div>
        </a>
    </div>
    <div class="col-xl-3 col-md-6 mb-3">
        <a href="?stock=critico" class="text-decoration-none">
            <div class="card metric-card bg-gradient-danger text-white hover-lift">
                <div class="card-body text-center py-4">
                    <div class="metric-icon mb-2">🚨</div>
                    <h4 class="metric-value fw-bold"><?= number_format($stats['stock_critico'], 0, '', '.') ?></h4>
                    <small class="metric-label">Stock Crítico</small>
                    <div class="mt-2">
                        <small class="opacity-75">🚨 Ver críticos</small>
                    </div>
                </div>
            </div>
        </a>
    </div>
    <div class="col-xl-3 col-md-6 mb-3">
        <a href="?stock=sin_stock" class="text-decoration-none">
            <div class="card metric-card bg-gradient-dark text-white hover-lift">
                <div class="card-body text-center py-4">
                    <div class="metric-icon mb-2">📭</div>
                    <h4 class="metric-value fw-bold"><?= number_format($stats['sin_stock'], 0, '', '.') ?></h4>
                    <small class="metric-label">Sin Stock</small>
                    <div class="mt-2">
                        <small class="opacity-75">📭 Ver agotados</small>
                    </div>
                </div>
            </div>
        </a>
    </div>
</div>

<!-- Formulario de Búsqueda y Filtros Mejorado -->
<div class="card border-0 shadow-sm mb-4">
    <div class="card-header bg-gradient-light py-3">
        <h6 class="mb-0">🔍 Búsqueda y Filtros Avanzados</h6>
    </div>
    <div class="card-body">
        <form method="get" class="row g-3" id="formBusqueda">
            <div class="col-md-4">
                <label class="form-label fw-semibold">Buscar por código o descripción</label>
                <input type="text" name="q" value="<?= htmlspecialchars($busqueda) ?>" 
                       class="form-control rounded-pill" placeholder="Ej: SARGA, COLCHÓN, etc." 
                       id="inputBusqueda">
            </div>
            <div class="col-md-3">
                <label class="form-label fw-semibold">Filtrar por categoría</label>
                <select name="categoria" class="form-select rounded-pill" id="selectCategoria">
                    <option value="">-- Todas las categorías --</option>
                    <?php foreach ($categorias as $cat): ?>
                        <option value="<?= $cat['id'] ?>" 
                                <?= ($categoria_filtro == $cat['id']) ? 'selected' : '' ?>>
                            <?= htmlspecialchars($cat['nombre']) ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div class="col-md-3">
                <label class="form-label fw-semibold">Estado de Stock</label>
                <select name="stock" class="form-select rounded-pill" id="selectStock">
                    <option value="">-- Todos los estados --</option>
                    <option value="stock_ok" <?= $filtro_stock == 'stock_ok' ? 'selected' : '' ?>>✅ Stock OK</option>
                    <option value="bajo" <?= $filtro_stock == 'bajo' ? 'selected' : '' ?>>⚠️ Stock Bajo</option>
                    <option value="critico" <?= $filtro_stock == 'critico' ? 'selected' : '' ?>>🚨 Stock Crítico</option>
                    <option value="sin_stock" <?= $filtro_stock == 'sin_stock' ? 'selected' : '' ?>>📭 Sin Stock</option>
                </select>
            </div>
            <div class="col-md-2 d-flex align-items-end">
                <button type="submit" class="btn btn-primary w-100 rounded-pill py-2">
                    <i class="bi bi-search me-2"></i>Buscar
                </button>
            </div>
        </form>
        
        <!-- Filtros activos -->
        <?php
        $filtrosActivos = [];
        if (!empty($busqueda)) $filtrosActivos[] = "Búsqueda: \"$busqueda\"";
        if (!empty($categoria_filtro)) {
            $catNombre = array_column($categorias, 'nombre', 'id')[$categoria_filtro] ?? '';
            $filtrosActivos[] = "Categoría: $catNombre";
        }
        if (!empty($filtro_stock)) {
            $estadosStock = [
                'stock_ok' => '✅ Stock OK',
                'bajo' => '⚠️ Stock Bajo', 
                'critico' => '🚨 Stock Crítico',
                'sin_stock' => '📭 Sin Stock'
            ];
            $filtrosActivos[] = $estadosStock[$filtro_stock] ?? '';
        }
        ?>
        
        <?php if (!empty($filtrosActivos)): ?>
            <div class="mt-4 p-3 bg-light rounded">
                <small class="text-muted fw-semibold">Filtros activos:</small>
                <?php foreach ($filtrosActivos as $filtro): ?>
                    <span class="badge bg-secondary rounded-pill me-1"><?= $filtro ?></span>
                <?php endforeach; ?>
                <a href="listar.php" class="btn btn-sm btn-outline-secondary rounded-pill ms-2">
                    <i class="bi bi-x-circle me-1"></i>Limpiar filtros
                </a>
            </div>
        <?php endif; ?>
    </div>
</div>

<!-- Botones de Acción Mejorados -->
<div class="d-flex justify-content-between align-items-center mb-4">
    <div class="d-flex gap-2">
        <a href="/inventario/cintas/agregar.php" class="btn btn-success rounded-pill px-4">
            <i class="bi bi-plus-circle me-2"></i>➕ Agregar Producto
        </a>
        <?php if ($_SESSION['usuario']['rol'] === 'admin'): ?>
            <a href="/inventario/categorias/agregar.php" class="btn btn-outline-primary rounded-pill px-4">
                <i class="bi bi-tags me-2"></i>📂 Categorías
            </a>
        <?php endif; ?>
    </div>
    <div class="text-muted">
        <span class="badge bg-primary rounded-pill p-2" id="contadorResultados">
            <i class="bi bi-box-seam me-1"></i>
            Mostrando <strong><?= count($cintas) ?></strong> de <strong><?= number_format($totalProductos, 0, '', '.') ?></strong> productos
        </span>
        <?php if ($totalPaginas > 1): ?>
            <span class="badge bg-secondary rounded-pill p-2 ms-2">
                Página <strong><?= $page ?></strong> de <strong><?= $totalPaginas ?></strong>
            </span>
        <?php endif; ?>
    </div>
</div>


<!-- Sistema de Exportación Rápida -->
<div class="card border-0 shadow-sm mb-4">
    <div class="card-header bg-gradient-success text-white py-3">
        <h6 class="mb-0">📤 Exportar Datos</h6>
    </div>
    <div class="card-body">
        <div class="row g-3">
            <div class="col-md-4">
                <div class="d-grid">
                    <a href="/inventario/exportar.php?tipo=inventario&formato=excel" 
                       class="btn btn-outline-success rounded-pill">
                        <i class="bi bi-file-earmark-excel me-2"></i>
                        Exportar Inventario (Excel)
                    </a>
                </div>
            </div>
            <div class="col-md-4">
                <div class="d-grid">
                    <a href="/inventario/exportar.php?tipo=movimientos&formato=excel" 
                       class="btn btn-outline-info rounded-pill">
                        <i class="bi bi-file-earmark-text me-2"></i>
                        Exportar Movimientos (Excel)
                    </a>
                </div>
            </div>
            <div class="col-md-4">
                <div class="dropdown">
                    <button class="btn btn-outline-warning rounded-pill w-100 dropdown-toggle" 
                            type="button" data-bs-toggle="dropdown">
                        <i class="bi bi-download me-2"></i>
                        Más Opciones
                    </button>
                    <ul class="dropdown-menu">
                        <li>
                            <a class="dropdown-item" href="#" onclick="exportarVistaActual()">
                                <i class="bi bi-eye me-2"></i>
                                Exportar vista actual
                            </a>
                        </li>
                        <li>
                            <a class="dropdown-item" href="#" onclick="compartirEnlace()">
                                <i class="bi bi-share me-2"></i>
                                Compartir enlace
                            </a>
                        </li>
                        <li><hr class="dropdown-divider"></li>
                        <li>
                            <a class="dropdown-item" href="/inventario/reportes/generar.php">
                                <i class="bi bi-graph-up me-2"></i>
                                Reportes avanzados
                            </a>
                        </li>
                    </ul>
                </div>
            </div>
        </div>
        
        <!-- Exportación de vista actual -->
        <div class="mt-3 p-3 bg-light rounded d-none" id="panelExportacion">
            <h6 class="mb-2">📋 Vista Actual para Exportar</h6>
            <div class="row g-2">
                <div class="col-md-6">
                    <strong>Filtros aplicados:</strong>
                    <div id="filtrosExportar" class="small text-muted"></div>
                </div>
                <div class="col-md-6">
                    <strong>Productos visibles:</strong>
                    <div id="contadorExportar" class="small text-muted"></div>
                </div>
            </div>
            <div class="mt-2">
                <button class="btn btn-sm btn-success" onclick="descargarVistaActual()">
                    <i class="bi bi-download me-1"></i> Descargar vista actual
                </button>
                <button class="btn btn-sm btn-outline-secondary" onclick="cerrarPanelExportacion()">
                    Cerrar
                </button>
            </div>
        </div>
    </div>
</div>



<!-- Tabla de Productos Mejorada -->
<div class="card border-0 shadow-sm">
    <div class="card-header bg-gradient-light py-3">
        <h6 class="mb-0">📋 Lista de Productos</h6>
    </div>
    <div class="card-body p-0">
        <div class="table-responsive">
            <table class="table table-hover mb-0">
                <thead class="table-light">
                    <tr>
                        <th class="border-0">Categoría</th>
                        <th class="border-0">Código</th>
                        <th class="border-0">Descripción</th>
                        <th class="border-0">Unidad</th>
                        <th class="border-0 text-end">Precio Unitario</th>
                        <th class="border-0 text-end">Stock</th>
                        <th class="border-0 text-end">Stock Mínimo</th>
                        <th class="border-0 text-end">Valor en Stock</th>
                        <th class="border-0 text-center">Acciones</th>
                    </tr>
                </thead>
                <tbody id="resultadosTabla">
                    <?php if (empty($cintas)): ?>
                        <tr>
                            <td colspan="9" class="text-center py-5">
                                <div class="text-muted">
                                    <i class="bi bi-search display-1 opacity-25"></i>
                                    <h5 class="mt-3">No se encontraron productos</h5>
                                    <p>Intenta con otros términos de búsqueda o <a href="agregar.php">agrega un nuevo producto</a></p>
                                </div>
                            </td>
                        </tr>
                    <?php else: ?>
                        <?php foreach ($cintas as $cinta): ?>
                            <?php 
                            $claseAlerta = getClaseAlerta($cinta['stock'], $cinta['stock_minimo']);
                            $valorStock = $cinta['stock'] * $cinta['precio_unitario'];
                            ?>
                            <tr class="<?= $claseAlerta ?> align-middle producto-fila" 
                                data-codigo="<?= htmlspecialchars($cinta['codigo']) ?>" 
                                data-descripcion="<?= htmlspecialchars($cinta['descripcion']) ?>">
                                <td>
                                    <span class="badge bg-secondary rounded-pill"><?= htmlspecialchars($cinta['categoria']) ?></span>
                                </td>
                                <td>
                                    <strong class="text-primary"><?= htmlspecialchars($cinta['codigo']) ?></strong>
                                </td>
                                <td><?= htmlspecialchars($cinta['descripcion']) ?></td>
                                <td>
                                    <span class="badge bg-light text-dark border rounded-pill"><?= traducirUnidad($cinta['unidad']) ?></span>
                                </td>
                                <td class="text-end">
                                    <strong class="text-success">$<?= number_format($cinta['precio_unitario'], 0, '', '.') ?></strong>
                                    <small class="text-muted d-block">CLP</small>
                                </td>
                                <td class="text-end">
                                    <span class="fw-bold <?= $cinta['stock'] < $cinta['stock_minimo'] ? 'text-danger' : 'text-dark' ?>">
                                        <?= number_format($cinta['stock'], 0, '', '.') ?>
                                    </span>
                                    <small class="text-muted d-block"><?= traducirUnidad($cinta['unidad']) ?></small>
                                </td>
                                <td class="text-end">
                                    <span class="<?= $cinta['stock'] < $cinta['stock_minimo'] ? 'text-danger fw-bold' : 'text-muted' ?>">
                                        <?= number_format($cinta['stock_minimo'], 0, '', '.') ?>
                                    </span>
                                    <small class="text-muted d-block"><?= traducirUnidad($cinta['unidad']) ?></small>
                                </td>
                                <td class="text-end">
                                    <strong class="text-dark">$<?= number_format($valorStock, 0, '', '.') ?></strong>
                                    <small class="text-muted d-block">CLP</small>
                                </td>
                                <td class="text-center">
                                    <div class="btn-group btn-group-sm" role="group">
                                        <a href="/inventario/movimientos/ingreso.php?cinta_id=<?= $cinta['id'] ?>" 
                                           class="btn btn-success rounded-pill px-3" title="Registrar ingreso">
                                            <i class="bi bi-plus-lg"></i>
                                        </a>
                                        <a href="/inventario/movimientos/salida.php?cinta_id=<?= $cinta['id'] ?>" 
                                           class="btn btn-danger rounded-pill px-3" title="Registrar salida">
                                            <i class="bi bi-dash-lg"></i>
                                        </a>
                                        <?php if ($_SESSION['usuario']['rol'] === 'admin'): ?>
                                            <a href="/inventario/cintas/editar.php?id=<?= $cinta['id'] ?>" 
                                               class="btn btn-warning rounded-pill px-3" title="Editar producto">
                                                <i class="bi bi-pencil"></i>
                                            </a>
                                        <?php endif; ?>
                                    </div>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>

<!-- Paginación Mejorada -->
<?php if ($totalPaginas > 1): ?>
<div class="card border-0 shadow-sm mt-4">
    <div class="card-body">
        <nav aria-label="Paginación">
            <ul class="pagination justify-content-center mb-0">
                <!-- Botón Anterior -->
                <li class="page-item <?= $page <= 1 ? 'disabled' : '' ?>">
                    <a class="page-link rounded-pill mx-1" href="?<?= http_build_query(array_merge($_GET, ['page' => $page - 1])) ?>">
                        <i class="bi bi-chevron-left me-1"></i>Anterior
                    </a>
                </li>
                
                <!-- Primera página -->
                <?php if ($page > 3): ?>
                    <li class="page-item">
                        <a class="page-link rounded-circle mx-1" href="?<?= http_build_query(array_merge($_GET, ['page' => 1])) ?>">1</a>
                    </li>
                    <?php if ($page > 4): ?>
                        <li class="page-item disabled">
                            <span class="page-link rounded-circle mx-1">...</span>
                        </li>
                    <?php endif; ?>
                <?php endif; ?>
                
                <!-- Páginas alrededor de la actual -->
                <?php for ($i = max(1, $page - 2); $i <= min($totalPaginas, $page + 2); $i++): ?>
                    <li class="page-item <?= ($i == $page) ? 'active' : '' ?>">
                        <a class="page-link rounded-circle mx-1" href="?<?= http_build_query(array_merge($_GET, ['page' => $i])) ?>">
                            <?= $i ?>
                        </a>
                    </li>
                <?php endfor; ?>
                
                <!-- Última página -->
                <?php if ($page < $totalPaginas - 2): ?>
                    <?php if ($page < $totalPaginas - 3): ?>
                        <li class="page-item disabled">
                            <span class="page-link rounded-circle mx-1">...</span>
                        </li>
                    <?php endif; ?>
                    <li class="page-item">
                        <a class="page-link rounded-circle mx-1" href="?<?= http_build_query(array_merge($_GET, ['page' => $totalPaginas])) ?>">
                            <?= $totalPaginas ?>
                        </a>
                    </li>
                <?php endif; ?>
                
                <!-- Botón Siguiente -->
                <li class="page-item <?= $page >= $totalPaginas ? 'disabled' : '' ?>">
                    <a class="page-link rounded-pill mx-1" href="?<?= http_build_query(array_merge($_GET, ['page' => $page + 1])) ?>">
                        Siguiente<i class="bi bi-chevron-right ms-1"></i>
                    </a>
                </li>
            </ul>
        </nav>

        <div class="text-center text-muted mt-3">
            <small>
                Página <strong><?= $page ?></strong> de <strong><?= $totalPaginas ?></strong> 
                • <?= number_format($totalProductos, 0, '', '.') ?> productos en total
                • <?= $limit ?> productos por página
            </small>
        </div>
    </div>
</div>
<?php endif; ?>

<!-- Sistema de Notificaciones -->
<div class="position-fixed top-0 end-0 p-3" style="z-index: 1050">
    <div id="liveToast" class="toast" role="alert">
        <div class="toast-header">
            <i class="bi bi-bell-fill text-primary me-2"></i>
            <strong class="me-auto">Sistema de Inventario</strong>
            <small>Ahora</small>
            <button type="button" class="btn-close" data-bs-dismiss="toast"></button>
        </div>
        <div class="toast-body" id="toastMessage">
            ¡Bienvenido al sistema!
        </div>
    </div>
</div>

<style>
.metric-card {
    border: none;
    border-radius: 15px;
    transition: all 0.3s ease;
}

.metric-icon {
    font-size: 1.8rem;
    opacity: 0.9;
}

.metric-value {
    font-size: 1.5rem;
    margin-bottom: 0.5rem;
}

.metric-label {
    font-size: 0.8rem;
    opacity: 0.9;
}

.hover-lift:hover {
    transform: translateY(-5px);
    box-shadow: 0 8px 25px rgba(0,0,0,0.15);
}

.bg-gradient-primary {
    background: linear-gradient(135deg, #007bff, #0056b3);
}

.bg-gradient-warning {
    background: linear-gradient(135deg, #ffc107, #e0a800);
}

.bg-gradient-danger {
    background: linear-gradient(135deg, #dc3545, #c82333);
}

.bg-gradient-dark {
    background: linear-gradient(135deg, #343a40, #23272b);
}

.bg-gradient-light {
    background: linear-gradient(135deg, #f8f9fa, #e9ecef);
}

.rounded-pill {
    border-radius: 50rem !important;
}

.rounded-circle {
    border-radius: 50% !important;
}

.btn-group-sm > .btn {
    border-radius: 50rem !important;
}

.table-hover tbody tr:hover {
    background-color: rgba(0, 123, 255, 0.05);
    cursor: pointer;
}

.border-0 {
    border: 0 !important;
}

.shadow-sm {
    box-shadow: 0 0.125rem 0.25rem rgba(0, 0, 0, 0.075) !important;
}

.producto-fila {
    transition: all 0.2s ease;
}

.producto-fila:hover {
    transform: scale(1.01);
    box-shadow: 0 2px 8px rgba(0,0,0,0.1);
}
</style>

<script>
// Sistema de Notificaciones
function mostrarNotificacion(mensaje, tipo = 'info') {
    const toast = new bootstrap.Toast(document.getElementById('liveToast'));
    const toastBody = document.getElementById('toastMessage');
    
    // Limpiar clases anteriores
    toastBody.className = 'toast-body';
    
    // Agregar clase según tipo
    if (tipo === 'success') {
        toastBody.classList.add('text-success');
    } else if (tipo === 'warning') {
        toastBody.classList.add('text-warning');
    } else if (tipo === 'danger') {
        toastBody.classList.add('text-danger');
    } else {
        toastBody.classList.add('text-info');
    }
    
    toastBody.innerHTML = mensaje;
    toast.show();
}

// Sistema de Productos Recientes
function guardarProductoReciente(codigo, descripcion) {
    let recientes = JSON.parse(localStorage.getItem('productos_recientes') || '[]');
    
    // Evitar duplicados
    recientes = recientes.filter(item => item.codigo !== codigo);
    
    // Agregar al inicio
    recientes.unshift({
        codigo: codigo,
        descripcion: descripcion,
        timestamp: Date.now()
    });
    
    // Mantener solo los últimos 10
    recientes = recientes.slice(0, 10);
    
    localStorage.setItem('productos_recientes', JSON.stringify(recientes));
    mostrarNotificacion(`✅ "${codigo" agregado a recientes`, 'success');
}

// Búsqueda en Tiempo Real
document.addEventListener('DOMContentLoaded', function() {
    const searchInput = document.getElementById('inputBusqueda');
    const categoriaSelect = document.getElementById('selectCategoria');
    const stockSelect = document.getElementById('selectStock');
    const resultadosContainer = document.getElementById('resultadosTabla');
    const contadorResultados = document.getElementById('contadorResultados');
    const formBusqueda = document.getElementById('formBusqueda');
    
    function actualizarResultados(data) {
        if (data.length === 0) {
            resultadosContainer.innerHTML = `
                <tr>
                    <td colspan="9" class="text-center py-5">
                        <div class="text-muted">
                            <i class="bi bi-search display-1 opacity-25"></i>
                            <h5 class="mt-3">No se encontraron productos</h5>
                            <p>Intenta con otros términos de búsqueda</p>
                        </div>
                    </td>
                </tr>
            `;
            contadorResultados.innerHTML = `
                <i class="bi bi-box-seam me-1"></i>
                Mostrando <strong>0</strong> de <strong>0</strong> productos
            `;
            return;
        }
        
        let html = '';
        data.forEach(item => {
            const alertClass = item.alerta_stock === 'warning' ? 'table-warning' : 
                              item.alerta_stock === 'danger' ? 'table-danger' : '';
            
            html += `
                <tr class="${alertClass} align-middle producto-fila" 
                    data-codigo="${item.codigo}" 
                    data-descripcion="${item.descripcion}">
                    <td>
                        <span class="badge bg-secondary rounded-pill">${item.categoria}</span>
                    </td>
                    <td>
                        <strong class="text-primary">${item.codigo}</strong>
                    </td>
                    <td>${item.descripcion}</td>
                    <td>
                        <span class="badge bg-light text-dark border rounded-pill">${item.unidad}</span>
                    </td>
                    <td class="text-end">
                        <strong class="text-success">$${item.precio_unitario}</strong>
                        <small class="text-muted d-block">CLP</small>
                    </td>
                    <td class="text-end">
                        <span class="fw-bold ${item.alerta_stock === 'danger' ? 'text-danger' : 'text-dark'}">
                            ${item.stock}
                        </span>
                        <small class="text-muted d-block">${item.unidad}</small>
                    </td>
                    <td class="text-end">
                        <span class="${item.alerta_stock !== 'success' ? 'text-danger fw-bold' : 'text-muted'}">
                            ${item.stock_minimo}
                        </span>
                        <small class="text-muted d-block">${item.unidad}</small>
                    </td>
                    <td class="text-end">
                        <strong class="text-dark">$${item.valor_stock}</strong>
                        <small class="text-muted d-block">CLP</small>
                    </td>
                    <td class="text-center">
                        <div class="btn-group btn-group-sm" role="group">
                            <a href="/inventario/movimientos/ingreso.php?cinta_id=${item.id}" 
                               class="btn btn-success rounded-pill px-3" title="Registrar ingreso">
                                <i class="bi bi-plus-lg"></i>
                            </a>
                            <a href="/inventario/movimientos/salida.php?cinta_id=${item.id}" 
                               class="btn btn-danger rounded-pill px-3" title="Registrar salida">
                                <i class="bi bi-dash-lg"></i>
                            </a>
                            <?php if ($_SESSION['usuario']['rol'] === 'admin'): ?>
                                <a href="/inventario/cintas/editar.php?id=${item.id}" 
                                   class="btn btn-warning rounded-pill px-3" title="Editar producto">
                                    <i class="bi bi-pencil"></i>
                                </a>
                            <?php endif; ?>
                        </div>
                    </td>
                </tr>
            `;
        });
        
        resultadosContainer.innerHTML = html;
        contadorResultados.innerHTML = `
            <i class="bi bi-box-seam me-1"></i>
            Mostrando <strong>${data.length}</strong> de <strong>${data.length}</strong> productos
        `;
        
        // Agregar event listeners para guardar como reciente
        document.querySelectorAll('.producto-fila').forEach(fila => {
            fila.addEventListener('click', function(e) {
                // No guardar si se hizo clic en un botón de acción
                if (e.target.closest('a') || e.target.closest('button')) {
                    return;
                }
                
                const codigo = this.getAttribute('data-codigo');
                const descripcion = this.getAttribute('data-descripcion');
                guardarProductoReciente(codigo, descripcion);
            });
        });
        
        // Mostrar notificación de resultados
        if (data.length > 0) {
            mostrarNotificacion(`✅ Encontrados ${data.length} producto(s)`, 'success');
        }
    }
    
    function realizarBusqueda() {
        const busqueda = searchInput.value;
        const categoria = categoriaSelect.value;
        const stock = stockSelect.value;
        
        // Mostrar loading
        resultadosContainer.innerHTML = `
            <tr>
                <td colspan="9" class="text-center py-4">
                    <div class="spinner-border text-primary" role="status">
                        <span class="visually-hidden">Buscando...</span>
                    </div>
                    <p class="mt-2 text-muted">Buscando productos...</p>
                </td>
            </tr>
        `;
        
        fetch(`/inventario/cintas/buscar_ajax.php?q=${encodeURIComponent(busqueda)}&categoria=${categoria}&stock=${stock}`)
            .then(response => {
                if (!response.ok) throw new Error('Error en la búsqueda');
                return response.json();
            })
            .then(data => {
                actualizarResultados(data);
            })
            .catch(error => {
                console.error('Error:', error);
                resultadosContainer.innerHTML = `
                    <tr>
                        <td colspan="9" class="text-center py-4 text-danger">
                            <i class="bi bi-exclamation-triangle"></i>
                            <p>Error al realizar la búsqueda</p>
                            <button class="btn btn-sm btn-outline-primary mt-2" onclick="location.reload()">
                                Recargar página
                            </button>
                        </td>
                    </tr>
                `;
                mostrarNotificacion('❌ Error en la búsqueda automática', 'danger');
            });
    }
    
    // Debounce para evitar muchas requests
    let timeout;
    [searchInput, categoriaSelect, stockSelect].forEach(element => {
        element.addEventListener('input', function() {
            clearTimeout(timeout);
            timeout = setTimeout(realizarBusqueda, 500);
        });
    });
    
    // Prevenir envío del formulario tradicional
    formBusqueda.addEventListener('submit', function(e) {
        e.preventDefault();
        realizarBusqueda();
    });
    
    // Búsqueda inicial si hay parámetros
    if (searchInput.value || categoriaSelect.value || stockSelect.value) {
        realizarBusqueda();
    }
    
    // Agregar event listeners a las filas existentes (para búsqueda tradicional)
    document.querySelectorAll('.producto-fila').forEach(fila => {
        fila.addEventListener('click', function(e) {
            if (e.target.closest('a') || e.target.closest('button')) {
                return;
            }
            
            const codigo = this.getAttribute('data-codigo');
            const descripcion = this.getAttribute('data-descripcion');
            guardarProductoReciente(codigo, descripcion);
        });
    });
});

// Notificación de bienvenida
document.addEventListener('DOMContentLoaded', function() {
    setTimeout(() => {
        mostrarNotificacion('🔍 Búsqueda en tiempo real activada - Escribe para filtrar', 'info');
    }, 1500);
});
</script>

<?php include __DIR__ . '/../includes/footer.php'; ?>

<script>
// Sistema de Exportación Rápida
function exportarVistaActual() {
    const busqueda = document.getElementById('inputBusqueda').value;
    const categoria = document.getElementById('selectCategoria').value;
    const stock = document.getElementById('selectStock').value;
    
    // Mostrar panel de exportación
    const panel = document.getElementById('panelExportacion');
    panel.classList.remove('d-none');
    
    // Actualizar información
    document.getElementById('filtrosExportar').innerHTML = `
        ${busqueda ? `Búsqueda: "${busqueda}"<br>` : ''}
        ${categoria ? `Categoría: ${document.getElementById('selectCategoria').options[document.getElementById('selectCategoria').selectedIndex].text}<br>` : ''}
        ${stock ? `Estado: ${document.getElementById('selectStock').options[document.getElementById('selectStock').selectedIndex].text}` : ''}
        ${!busqueda && !categoria && !stock ? 'Sin filtros aplicados' : ''}
    `;
    
    const contador = document.getElementById('contadorResultados').textContent;
    document.getElementById('contadorExportar').textContent = contador;
}

function cerrarPanelExportacion() {
    document.getElementById('panelExportacion').classList.add('d-none');
}

function descargarVistaActual() {
    const busqueda = document.getElementById('inputBusqueda').value;
    const categoria = document.getElementById('selectCategoria').value;
    const stock = document.getElementById('selectStock').value;
    
    let url = `/inventario/exportar.php?tipo=inventario&formato=excel`;
    
    if (busqueda) url += `&busqueda=${encodeURIComponent(busqueda)}`;
    if (categoria) url += `&categoria=${categoria}`;
    if (stock) url += `&stock=${stock}`;
    
    window.open(url, '_blank');
    mostrarNotificacion('✅ Descarga de vista actual iniciada', 'success');
}

function compartirEnlace() {
    const url = new URL(window.location.href);
    const enlace = url.toString();
    
    // Crear enlace temporal para copiar
    const tempInput = document.createElement('input');
    tempInput.value = enlace;
    document.body.appendChild(tempInput);
    tempInput.select();
    document.execCommand('copy');
    document.body.removeChild(tempInput);
    
    mostrarNotificacion('🔗 Enlace copiado al portapapeles', 'success');
}

// Exportación rápida con atajos
document.addEventListener('keydown', function(e) {
    if (e.ctrlKey && e.key === 'e') {
        e.preventDefault();
        exportarVistaActual();
    }
});
</script>