<?php
// Configuración de seguridad de sesión - DEBE IR ANTES de session_start()
ini_set('session.cookie_httponly', 1);
ini_set('session.cookie_secure', 1);
ini_set('session.use_strict_mode', 1);

session_start();

if (!isset($_SESSION['usuario'])) {
    // Guardar la URL actual para redirigir después del login
    $_SESSION['redirect_url'] = $_SERVER['REQUEST_URI'];
    header("Location: /inventario/auth/login.php");
    exit;
}

// Verificar inactividad (30 minutos)
$inactivity_limit = 30 * 60; // 30 minutos en segundos
if (isset($_SESSION['last_activity']) && (time() - $_SESSION['last_activity']) > $inactivity_limit) {
    session_unset();
    session_destroy();
    header("Location: /inventario/auth/login.php?timeout=1");
    exit;
}

// Actualizar timestamp de actividad
$_SESSION['last_activity'] = time();

// Regenerar ID de sesión periódicamente para prevenir fixation
if (!isset($_SESSION['created'])) {
    $_SESSION['created'] = time();
} else if (time() - $_SESSION['created'] > 1800) { // 30 minutos
    session_regenerate_id(true);
    $_SESSION['created'] = time();
}

// Validar que el usuario aún existe en la base de datos y está activo
if (!isset($_SESSION['user_validated']) || (time() - $_SESSION['user_validated']) > 300) {
    require __DIR__ . '/../config/db.php';
    
    try {
        $stmt = $pdo->prepare("SELECT id, usuario, rol, activo FROM usuarios WHERE id = ? AND activo = 1");
        $stmt->execute([$_SESSION['usuario']['id']]);
        $user = $stmt->fetch();
        
        if (!$user) {
            // Registrar intento de acceso con usuario inactivo
            error_log("Intento de acceso con usuario inactivo o eliminado: " . $_SESSION['usuario']['id']);
            session_unset();
            session_destroy();
            header("Location: /inventario/auth/login.php?invalid=1");
            exit;
        }
        
        // Actualizar información de sesión si hay cambios
        $_SESSION['usuario']['usuario'] = $user['usuario'];
        $_SESSION['usuario']['rol'] = $user['rol'];
        $_SESSION['user_validated'] = time();
        
    } catch (Exception $e) {
        error_log("Error validando usuario: " . $e->getMessage());
        // En caso de error, permitir continuar pero registrar
        $_SESSION['user_validated'] = time();
    }
}

// Verificación de permisos basada en roles
function tienePermiso($rolRequerido) {
    $rolUsuario = $_SESSION['usuario']['rol'] ?? '';
    
    $jerarquia = [
        'admin' => 3,
        'operador' => 2, 
        'auditor' => 1
    ];
    
    $nivelUsuario = $jerarquia[$rolUsuario] ?? 0;
    $nivelRequerido = $jerarquia[$rolRequerido] ?? 0;
    
    return $nivelUsuario >= $nivelRequerido;
}

// Función para verificar permisos específicos
function puede($accion) {
    $rol = $_SESSION['usuario']['rol'] ?? '';
    
    $permisos = [
        'admin' => ['crear', 'editar', 'eliminar', 'ver', 'reportes', 'usuarios'],
        'operador' => ['crear', 'editar', 'ver', 'reportes'],
        'auditor' => ['ver', 'reportes']
    ];
    
    return in_array($accion, $permisos[$rol] ?? []);
}

// Protección contra CSRF
if (!isset($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

function csrf_token() {
    return $_SESSION['csrf_token'];
}

function verificar_csrf($token) {
    return hash_equals($_SESSION['csrf_token'], $token);
}
?>