<?php
require __DIR__ . '/../includes/auth_check.php';
require __DIR__ . '/../config/db.php';
include __DIR__ . '/../includes/header.php';

// Función para traducir unidad
function traducirUnidad($u) {
    switch ($u) {
        case 'm': return 'Metros';
        case 'kg': return 'Kilos';
        case 'u': return 'Unidades';
        default: return $u;
    }
}

// Parámetros de filtro
$filtro_tipo = $_GET['tipo'] ?? 'todos';
$filtro_categoria = $_GET['categoria'] ?? 'todas';
$filtro_producto = $_GET['producto'] ?? '';
$filtro_fecha_inicio = $_GET['fecha_inicio'] ?? '';
$filtro_fecha_fin = $_GET['fecha_fin'] ?? '';
$filtro_documento = $_GET['documento'] ?? '';

// Paginación
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$limit = 50;
$offset = ($page - 1) * $limit;

// Construir consulta con filtros
$whereConditions = [];
$params = [];

// Filtro por tipo
if ($filtro_tipo !== 'todos') {
    $whereConditions[] = "m.tipo = :tipo";
    $params[':tipo'] = $filtro_tipo;
}

// Filtro por categoría
if ($filtro_categoria !== 'todas') {
    $whereConditions[] = "cat.id = :categoria_id";
    $params[':categoria_id'] = $filtro_categoria;
}

// Filtro por producto
if (!empty($filtro_producto)) {
    $whereConditions[] = "(c.codigo LIKE :producto OR c.descripcion LIKE :producto)";
    $params[':producto'] = "%$filtro_producto%";
}

// Filtro por fechas
if (!empty($filtro_fecha_inicio)) {
    $whereConditions[] = "DATE(m.fecha) >= :fecha_inicio";
    $params[':fecha_inicio'] = $filtro_fecha_inicio;
}

if (!empty($filtro_fecha_fin)) {
    $whereConditions[] = "DATE(m.fecha) <= :fecha_fin";
    $params[':fecha_fin'] = $filtro_fecha_fin;
}

// Filtro por documento
if (!empty($filtro_documento)) {
    $whereConditions[] = "m.documento LIKE :documento";
    $params[':documento'] = "%$filtro_documento%";
}

// Solo productos activos
$whereConditions[] = "c.activo = 1";

$whereClause = '';
if (!empty($whereConditions)) {
    $whereClause = 'WHERE ' . implode(' AND ', $whereConditions);
}

// Consulta para contar total
$sqlCount = "
    SELECT COUNT(*)
    FROM movimientos m
    JOIN cintas c ON m.cinta_id = c.id
    LEFT JOIN categorias cat ON c.categoria_id = cat.id
    $whereClause
";

$stmtCount = $pdo->prepare($sqlCount);
foreach ($params as $key => $value) {
    $stmtCount->bindValue($key, $value);
}
$stmtCount->execute();
$totalMovimientos = $stmtCount->fetchColumn();
$totalPaginas = ceil($totalMovimientos / $limit);

// Consulta principal con paginación
$sql = "
    SELECT m.*, c.codigo, c.descripcion, c.precio_unitario, c.unidad,
           cat.nombre as categoria_nombre,
           (m.cantidad * c.precio_unitario) as valor_movimiento,
           u.usuario as usuario_nombre
    FROM movimientos m
    JOIN cintas c ON m.cinta_id = c.id
    LEFT JOIN categorias cat ON c.categoria_id = cat.id
    LEFT JOIN usuarios u ON m.usuario_id = u.id
    $whereClause
    ORDER BY m.fecha DESC
    LIMIT :limit OFFSET :offset
";

$stmt = $pdo->prepare($sql);
foreach ($params as $key => $value) {
    $stmt->bindValue($key, $value);
}
$stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
$stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
$stmt->execute();
$movimientos = $stmt->fetchAll();

// Obtener categorías para el filtro
$categorias = $pdo->query("SELECT * FROM categorias ORDER BY nombre")->fetchAll();

// Obtener productos activos para el filtro
$productos = $pdo->query("SELECT id, codigo, descripcion FROM cintas WHERE activo = 1 ORDER BY codigo")->fetchAll();

// Estadísticas de movimientos
$sqlStats = "
    SELECT 
        COUNT(*) as total,
        SUM(CASE WHEN tipo = 'ingreso' THEN 1 ELSE 0 END) as ingresos,
        SUM(CASE WHEN tipo = 'salida' THEN 1 ELSE 0 END) as salidas,
        SUM(CASE WHEN tipo = 'ingreso' THEN cantidad ELSE 0 END) as cantidad_ingresos,
        SUM(CASE WHEN tipo = 'salida' THEN cantidad ELSE 0 END) as cantidad_salidas,
        SUM(CASE WHEN tipo = 'ingreso' THEN cantidad * c.precio_unitario ELSE 0 END) as valor_ingresos,
        SUM(CASE WHEN tipo = 'salida' THEN cantidad * c.precio_unitario ELSE 0 END) as valor_salidas
    FROM movimientos m
    JOIN cintas c ON m.cinta_id = c.id
    LEFT JOIN categorias cat ON c.categoria_id = cat.id
    $whereClause
";

$stmtStats = $pdo->prepare($sqlStats);
foreach ($params as $key => $value) {
    $stmtStats->bindValue($key, $value);
}
$stmtStats->execute();
$stats = $stmtStats->fetch();
?>
<h3 class="mb-4">📜 Historial de Movimientos</h3>

<!-- Tarjetas de Estadísticas -->
<div class="row mb-4">
    <div class="col-md-3">
        <div class="card bg-primary text-white">
            <div class="card-body text-center py-3">
                <h5><?= number_format($stats['total'] ?? 0, 0, '', '.') ?></h5>
                <small>Total Movimientos</small>
            </div>
        </div>
    </div>
    <div class="col-md-3">
        <div class="card bg-success text-white">
            <div class="card-body text-center py-3">
                <h5><?= number_format($stats['ingresos'] ?? 0, 0, '', '.') ?></h5>
                <small>Ingresos</small>
                <br>
                <small><?= formatearMoneda($stats['valor_ingresos'] ?? 0) ?></small>
            </div>
        </div>
    </div>
    <div class="col-md-3">
        <div class="card bg-danger text-white">
            <div class="card-body text-center py-3">
                <h5><?= number_format($stats['salidas'] ?? 0, 0, '', '.') ?></h5>
                <small>Salidas</small>
                <br>
                <small><?= formatearMoneda($stats['valor_salidas'] ?? 0) ?></small>
            </div>
        </div>
    </div>
    <div class="col-md-3">
        <div class="card bg-info text-white">
            <div class="card-body text-center py-3">
                <h5><?= number_format(($stats['cantidad_ingresos'] ?? 0) - ($stats['cantidad_salidas'] ?? 0), 0, '', '.') ?></h5>
                <small>Neto</small>
            </div>
        </div>
    </div>
</div>

<!-- Formulario de Filtros Avanzados -->
<div class="card mb-4">
    <div class="card-header bg-light">
        <h6 class="mb-0">🔍 Filtros Avanzados</h6>
    </div>
    <div class="card-body">
        <form method="get" class="row g-3">
            <!-- Fila 1 -->
            <div class="col-md-3">
                <label class="form-label">Tipo de Movimiento</label>
                <select name="tipo" class="form-select">
                    <option value="todos" <?= $filtro_tipo === 'todos' ? 'selected' : '' ?>>Todos los tipos</option>
                    <option value="ingreso" <?= $filtro_tipo === 'ingreso' ? 'selected' : '' ?>>Solo Ingresos</option>
                    <option value="salida" <?= $filtro_tipo === 'salida' ? 'selected' : '' ?>>Solo Salidas</option>
                </select>
            </div>
            <div class="col-md-3">
                <label class="form-label">Categoría</label>
                <select name="categoria" class="form-select">
                    <option value="todas" <?= $filtro_categoria === 'todas' ? 'selected' : '' ?>>Todas las categorías</option>
                    <?php foreach ($categorias as $cat): ?>
                        <option value="<?= $cat['id'] ?>" <?= $filtro_categoria == $cat['id'] ? 'selected' : '' ?>>
                            <?= htmlspecialchars($cat['nombre']) ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div class="col-md-3">
                <label class="form-label">Producto</label>
                <select name="producto" class="form-select">
                    <option value="">Todos los productos</option>
                    <?php foreach ($productos as $prod): ?>
                        <option value="<?= $prod['codigo'] ?>" <?= $filtro_producto == $prod['codigo'] ? 'selected' : '' ?>>
                            <?= htmlspecialchars($prod['codigo']) ?> - <?= htmlspecialchars($prod['descripcion']) ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div class="col-md-3">
                <label class="form-label">N° Documento</label>
                <input type="text" name="documento" value="<?= htmlspecialchars($filtro_documento) ?>" 
                       class="form-control" placeholder="Ej: F-001, B-123">
            </div>

            <!-- Fila 2 -->
            <div class="col-md-3">
                <label class="form-label">Fecha Desde</label>
                <input type="date" name="fecha_inicio" value="<?= htmlspecialchars($filtro_fecha_inicio) ?>" 
                       class="form-control">
            </div>
            <div class="col-md-3">
                <label class="form-label">Fecha Hasta</label>
                <input type="date" name="fecha_fin" value="<?= htmlspecialchars($filtro_fecha_fin) ?>" 
                       class="form-control">
            </div>
            <div class="col-md-6 d-flex align-items-end">
                <div class="d-flex gap-2 w-100">
                    <button type="submit" class="btn btn-primary flex-fill">
                        <i class="bi bi-funnel"></i> Aplicar Filtros
                    </button>
                    <a href="historial.php" class="btn btn-outline-secondary">
                        <i class="bi bi-arrow-clockwise"></i> Limpiar
                    </a>
                </div>
            </div>
        </form>

        <!-- Filtros activos -->
        <?php
        $filtrosActivos = [];
        if ($filtro_tipo !== 'todos') $filtrosActivos[] = "Tipo: " . ucfirst($filtro_tipo);
        if ($filtro_categoria !== 'todas') {
            $catNombre = array_column($categorias, 'nombre', 'id')[$filtro_categoria] ?? '';
            $filtrosActivos[] = "Categoría: " . $catNombre;
        }
        if (!empty($filtro_producto)) $filtrosActivos[] = "Producto: " . $filtro_producto;
        if (!empty($filtro_fecha_inicio)) $filtrosActivos[] = "Desde: " . $filtro_fecha_inicio;
        if (!empty($filtro_fecha_fin)) $filtrosActivos[] = "Hasta: " . $filtro_fecha_fin;
        if (!empty($filtro_documento)) $filtrosActivos[] = "Documento: " . $filtro_documento;
        ?>

        <?php if (!empty($filtrosActivos)): ?>
        <div class="mt-3 p-2 bg-light rounded">
            <small class="text-muted">Filtros activos:</small>
            <?php foreach ($filtrosActivos as $filtro): ?>
                <span class="badge bg-secondary me-1"><?= $filtro ?></span>
            <?php endforeach; ?>
        </div>
        <?php endif; ?>
    </div>
</div>

<!-- Información de resultados -->
<div class="d-flex justify-content-between align-items-center mb-3">
    <div>
        <span class="text-muted">
            Mostrando <strong><?= count($movimientos) ?></strong> de <strong><?= number_format($totalMovimientos, 0, '', '.') ?></strong> movimientos
        </span>
    </div>
    <div>
        <a href="/inventario/reportes/generar.php?periodo=personalizado&fecha_inicio=<?= urlencode($filtro_fecha_inicio) ?>&fecha_fin=<?= urlencode($filtro_fecha_fin) ?>" 
           class="btn btn-sm btn-success">
            <i class="bi bi-download"></i> Exportar
        </a>
    </div>
</div>

<!-- Tabla de Movimientos -->
<div class="table-responsive">
    <table class="table table-striped table-hover shadow-sm">
        <thead class="table-primary">
            <tr>
                <th>Fecha y Hora</th>
                <th>Tipo</th>
                <th>Categoría</th>
                <th>Código</th>
                <th>Descripción</th>
                <th>Cantidad</th>
                <th>Unidad</th>
                <th>Precio Unitario</th>
                <th>Valor Movimiento</th>
                <th>Documento</th>
                <th>Usuario</th>
                <th>Acciones</th>
            </tr>
        </thead>
        <tbody>
            <?php if (empty($movimientos)): ?>
                <tr>
                    <td colspan="12" class="text-center py-4">
                        <div class="text-muted">
                            <h5>📭 No se encontraron movimientos</h5>
                            <p>Intenta con otros filtros o ajusta los criterios de búsqueda</p>
                        </div>
                    </td>
                </tr>
            <?php else: ?>
                <?php foreach ($movimientos as $m): ?>
                    <?php 
                    $rowClass = ($m['tipo'] === 'ingreso') ? 'bg-light-green' : 'bg-light-red';
                    $iconoTipo = ($m['tipo'] === 'ingreso') ? '⬆️' : '⬇️';
                    $colorTipo = ($m['tipo'] === 'ingreso') ? 'success' : 'danger';
                    ?>
                    <tr class="<?= $rowClass ?>">
                        <td>
                            <small class="text-muted"><?= date('d/m/Y', strtotime($m['fecha'])) ?></small>
                            <br>
                            <small><?= date('H:i', strtotime($m['fecha'])) ?></small>
                        </td>
                        <td>
                            <span class="badge bg-<?= $colorTipo ?>">
                                <?= $iconoTipo ?> <?= ucfirst($m['tipo']) ?>
                            </span>
                        </td>
                        <td>
                            <span class="badge bg-secondary"><?= htmlspecialchars($m['categoria_nombre']) ?></span>
                        </td>
                        <td>
                            <strong><?= htmlspecialchars($m['codigo']) ?></strong>
                        </td>
                        <td>
                            <?= htmlspecialchars($m['descripcion']) ?>
                            <?php if (!empty($m['observaciones'])): ?>
                                <br>
                                <small class="text-muted" title="<?= htmlspecialchars($m['observaciones']) ?>">
                                    📝 <?= mb_strimwidth(htmlspecialchars($m['observaciones']), 0, 50, '...') ?>
                                </small>
                            <?php endif; ?>
                        </td>
                        <td class="text-end fw-bold">
                            <?= number_format($m['cantidad'], 0, '', '.') ?>
                        </td>
                        <td>
                            <span class="badge bg-light text-dark"><?= traducirUnidad($m['unidad']) ?></span>
                        </td>
                        <td class="text-end">
                            $<?= number_format($m['precio_unitario'], 0, '', '.') ?>
                            <small class="text-muted d-block">CLP</small>
                        </td>
                        <td class="text-end fw-bold text-<?= $colorTipo ?>">
                            $<?= number_format($m['valor_movimiento'], 0, '', '.') ?>
                            <small class="text-muted d-block">CLP</small>
                        </td>
                        <td>
                            <small class="text-uppercase"><?= htmlspecialchars($m['tipo_doc'] ?? '-') ?></small>
                            <br>
                            <code><?= htmlspecialchars($m['documento'] ?? '-') ?></code>
                        </td>
                        <td>
                            <small><?= htmlspecialchars($m['usuario_nombre'] ?? 'Sistema') ?></small>
                        </td>
                        <td>
                            <div class="btn-group btn-group-sm">
                                <a href="/inventario/movimientos/<?= $m['tipo'] ?>.php?cinta_id=<?= $m['cinta_id'] ?>" 
                                   class="btn btn-outline-<?= $colorTipo ?>" 
                                   title="Nuevo <?= $m['tipo'] ?>">
                                    <?= $iconoTipo ?>
                                </a>
                                <a href="/inventario/cintas/listar.php?q=<?= urlencode($m['codigo']) ?>" 
                                   class="btn btn-outline-info" 
                                   title="Ver producto">
                                    📦
                                </a>
                            </div>
                        </td>
                    </tr>
                <?php endforeach; ?>
            <?php endif; ?>
        </tbody>
    </table>
</div>

<!-- Paginación Mejorada -->
<?php if ($totalPaginas > 1): ?>
<nav aria-label="Paginación del historial">
    <ul class="pagination justify-content-center">
        <!-- Botón Anterior -->
        <li class="page-item <?= $page <= 1 ? 'disabled' : '' ?>">
            <a class="page-link" href="?<?= http_build_query(array_merge($_GET, ['page' => $page - 1])) ?>">
                ‹ Anterior
            </a>
        </li>
        
        <!-- Páginas -->
        <?php for ($i = 1; $i <= $totalPaginas; $i++): ?>
            <?php if ($i == 1 || $i == $totalPaginas || ($i >= $page - 2 && $i <= $page + 2)): ?>
                <li class="page-item <?= ($i == $page) ? 'active' : '' ?>">
                    <a class="page-link" href="?<?= http_build_query(array_merge($_GET, ['page' => $i])) ?>">
                        <?= $i ?>
                    </a>
                </li>
            <?php elseif ($i == $page - 3 || $i == $page + 3): ?>
                <li class="page-item disabled">
                    <span class="page-link">...</span>
                </li>
            <?php endif; ?>
        <?php endfor; ?>
        
        <!-- Botón Siguiente -->
        <li class="page-item <?= $page >= $totalPaginas ? 'disabled' : '' ?>">
            <a class="page-link" href="?<?= http_build_query(array_merge($_GET, ['page' => $page + 1])) ?>">
                Siguiente ›
            </a>
        </li>
    </ul>
</nav>

<div class="text-center text-muted mt-2">
    Página <?= $page ?> de <?= $totalPaginas ?> 
    • <?= number_format($totalMovimientos, 0, '', '.') ?> movimientos en total
</div>
<?php endif; ?>

<!-- Información de resumen cuando hay filtros -->
<?php if (!empty($filtrosActivos) && !empty($movimientos)): ?>
<div class="card mt-4">
    <div class="card-header bg-light">
        <h6 class="mb-0">📊 Resumen con Filtros Aplicados</h6>
    </div>
    <div class="card-body">
        <div class="row text-center">
            <div class="col-md-3">
                <strong class="text-success"><?= number_format($stats['ingresos'] ?? 0, 0, '', '.') ?></strong>
                <br>
                <small class="text-muted">Ingresos</small>
                <br>
                <small class="text-success"><?= formatearMoneda($stats['valor_ingresos'] ?? 0) ?></small>
            </div>
            <div class="col-md-3">
                <strong class="text-danger"><?= number_format($stats['salidas'] ?? 0, 0, '', '.') ?></strong>
                <br>
                <small class="text-muted">Salidas</small>
                <br>
                <small class="text-danger"><?= formatearMoneda($stats['valor_salidas'] ?? 0) ?></small>
            </div>
            <div class="col-md-3">
                <strong class="text-primary"><?= number_format($stats['total'] ?? 0, 0, '', '.') ?></strong>
                <br>
                <small class="text-muted">Total Movimientos</small>
            </div>
            <div class="col-md-3">
                <strong class="text-info"><?= number_format(($stats['cantidad_ingresos'] ?? 0) - ($stats['cantidad_salidas'] ?? 0), 0, '', '.') ?></strong>
                <br>
                <small class="text-muted">Cantidad Neta</small>
                <br>
                <small class="text-info"><?= formatearMoneda(($stats['valor_ingresos'] ?? 0) - ($stats['valor_salidas'] ?? 0)) ?></small>
            </div>
        </div>
    </div>
</div>
<?php endif; ?>

<?php
// Función auxiliar para formatear moneda
function formatearMoneda($valor) {
    return '$' . number_format($valor, 0, '', '.') . ' CLP';
}
?>

<?php include __DIR__ . '/../includes/footer.php'; ?>