<?php
require __DIR__ . '/../includes/auth_check.php';
require __DIR__ . '/../config/db.php';
include __DIR__ . '/../includes/header.php';

if ($_SESSION['usuario']['rol'] === 'auditor') {
    die("Acceso denegado.");
}

$cinta_id_preseleccionada = $_GET['cinta_id'] ?? null;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $cinta_id  = $_POST['cinta_id'];
    $cantidad  = (float)$_POST['cantidad'];
    $unidad    = $_POST['unidad'];
    $tipo_doc  = $_POST['tipo_doc'];
    $documento = $_POST['documento'];
    $observaciones = $_POST['observaciones'] ?? '';
    $usuario_id = $_SESSION['usuario']['id'];

    // Validar cantidad positiva
    if ($cantidad <= 0) {
        $mensaje = "❌ La cantidad debe ser mayor a cero.";
        $tipoMensaje = "danger";
    } else {
        try {
            $pdo->beginTransaction();
            
            // Obtener información del producto para auditoría
            $stmt = $pdo->prepare("SELECT codigo, descripcion, stock FROM cintas WHERE id = ? AND activo = 1");
            $stmt->execute([$cinta_id]);
            $producto = $stmt->fetch();
            
            if (!$producto) {
                throw new Exception("Producto no encontrado o inactivo.");
            }
            
            // Guardar movimiento
            $stmt = $pdo->prepare("INSERT INTO movimientos (cinta_id, tipo, cantidad, unidad, tipo_doc, documento, observaciones, usuario_id) 
                                   VALUES (?, 'ingreso', ?, ?, ?, ?, ?, ?)");
            $stmt->execute([$cinta_id, $cantidad, $unidad, $tipo_doc, $documento, $observaciones, $usuario_id]);
            
            // Actualizar stock
            $stmt = $pdo->prepare("UPDATE cintas SET stock = stock + ?, actualizado_en = CURRENT_TIMESTAMP WHERE id = ?");
            $stmt->execute([$cantidad, $cinta_id]);
            
            $pdo->commit();
            
            $mensaje = "✅ Ingreso registrado correctamente. ";
            $mensaje .= "Producto: <strong>" . htmlspecialchars($producto['codigo']) . "</strong> - ";
            $mensaje .= "Cantidad: <strong>" . number_format($cantidad, 0, '', '.') . " " . $unidad . "</strong> - ";
            $mensaje .= "Stock anterior: <strong>" . number_format($producto['stock'], 0, '', '.') . "</strong> - ";
            $mensaje .= "Nuevo stock: <strong>" . number_format($producto['stock'] + $cantidad, 0, '', '.') . "</strong>";
            $tipoMensaje = "success";
            
            // Limpiar formulario después de éxito
            $_POST = [];
            
        } catch (Exception $e) {
            $pdo->rollBack();
            $mensaje = "❌ Error al registrar ingreso: " . $e->getMessage();
            $tipoMensaje = "danger";
        }
    }
}

// Obtener productos activos ordenados por categoría y código
$cintas = $pdo->query("
    SELECT c.*, cat.nombre as categoria 
    FROM cintas c 
    LEFT JOIN categorias cat ON c.categoria_id = cat.id 
    WHERE c.activo = 1
    ORDER BY cat.nombre, c.codigo ASC
")->fetchAll();

// Agrupar productos por categoría para el select optimizado
$cintasPorCategoria = [];
foreach ($cintas as $c) {
    $cintasPorCategoria[$c['categoria']][] = $c;
}
?>
<h3 class="mb-4">➕ Registrar Ingreso</h3>

<?php if(isset($mensaje)): ?>
  <div class="alert alert-<?= $tipoMensaje ?? 'info' ?>"><?= $mensaje ?></div>
<?php endif; ?>

<form method="POST" class="card p-4 shadow-sm">
  <div class="mb-3">
    <label class="form-label">Producto <span class="text-danger">*</span></label>
    <select name="cinta_id" class="form-select" required id="select-cinta">
      <option value="">-- Seleccionar Producto --</option>
      <?php foreach($cintasPorCategoria as $categoria => $productos): ?>
        <optgroup label="📁 <?= htmlspecialchars($categoria) ?>">
          <?php foreach($productos as $c): ?>
            <option value="<?= $c['id'] ?>" 
                    data-stock="<?= $c['stock'] ?>"
                    data-unidad="<?= $c['unidad'] ?>"
                    data-precio="<?= $c['precio_unitario'] ?>"
                    data-stock-minimo="<?= $c['stock_minimo'] ?>"
                    <?= (($cinta_id_preseleccionada == $c['id']) || (isset($_POST['cinta_id']) && $_POST['cinta_id'] == $c['id'])) ? 'selected' : '' ?>>
              <?= $c['codigo'] ?> - <?= $c['descripcion'] ?> 
              (Stock: <?= number_format($c['stock'], 0, '', '.') ?> <?= $c['unidad'] ?>)
            </option>
          <?php endforeach; ?>
        </optgroup>
      <?php endforeach; ?>
    </select>
    <div class="form-text" id="stock-info"></div>
  </div>

  <div class="row">
    <div class="col-md-6">
      <div class="mb-3">
        <label class="form-label">Cantidad <span class="text-danger">*</span></label>
        <input type="number" step="0.01" min="0.01" name="cantidad" 
               value="<?= isset($_POST['cantidad']) ? htmlspecialchars($_POST['cantidad']) : '' ?>" 
               class="form-control" required id="input-cantidad"
               placeholder="Ej: 100, 50.5, 25.75">
        <div class="form-text" id="valor-total-info"></div>
      </div>
    </div>
    <div class="col-md-6">
      <div class="mb-3">
        <label class="form-label">Unidad de Medida <span class="text-danger">*</span></label>
        <select name="unidad" class="form-select" required id="select-unidad">
          <option value="m">Metros (m)</option>
          <option value="kg">Kilos (kg)</option>
          <option value="u">Unidades (u)</option>
        </select>
      </div>
    </div>
  </div>

  <div class="row">
    <div class="col-md-6">
      <div class="mb-3">
        <label class="form-label">Tipo de Documento <span class="text-danger">*</span></label>
        <select name="tipo_doc" class="form-select" required>
          <option value="factura" <?= (isset($_POST['tipo_doc']) && $_POST['tipo_doc'] == 'factura') ? 'selected' : '' ?>>Factura</option>
          <option value="boleta" <?= (isset($_POST['tipo_doc']) && $_POST['tipo_doc'] == 'boleta') ? 'selected' : '' ?>>Boleta</option>
          <option value="nota_venta" <?= (isset($_POST['tipo_doc']) && $_POST['tipo_doc'] == 'nota_venta') ? 'selected' : '' ?>>Nota de Venta</option>
          <option value="nota_despacho" <?= (isset($_POST['tipo_doc']) && $_POST['tipo_doc'] == 'nota_despacho') ? 'selected' : '' ?>>Nota de Despacho</option>
          <option value="interno" <?= (isset($_POST['tipo_doc']) && $_POST['tipo_doc'] == 'interno') ? 'selected' : '' ?>>Documento Interno</option>
          <option value="ajuste" <?= (isset($_POST['tipo_doc']) && $_POST['tipo_doc'] == 'ajuste') ? 'selected' : '' ?>>Ajuste de Inventario</option>
        </select>
      </div>
    </div>
    <div class="col-md-6">
      <div class="mb-3">
        <label class="form-label">Número de Documento <span class="text-danger">*</span></label>
        <input type="text" name="documento" 
               value="<?= isset($_POST['documento']) ? htmlspecialchars($_POST['documento']) : '' ?>" 
               class="form-control" required 
               placeholder="Ej: F-001, B-123, NV-456, etc.">
      </div>
    </div>
  </div>

  <div class="mb-3">
    <label class="form-label">Observaciones (Opcional)</label>
    <textarea name="observaciones" class="form-control" rows="3" 
              placeholder="Observaciones adicionales sobre este ingreso..."><?= isset($_POST['observaciones']) ? htmlspecialchars($_POST['observaciones']) : '' ?></textarea>
  </div>

  <div class="d-grid gap-2">
    <button type="submit" class="btn btn-success btn-lg py-2">
      <i class="bi bi-check-circle"></i> 💾 Registrar Ingreso
    </button>
    <a href="/inventario/cintas/listar.php" class="btn btn-outline-secondary">
      📦 Volver al Inventario
    </a>
  </div>
</form>

<!-- Información del producto seleccionado -->
<div class="card mt-4">
  <div class="card-header bg-light">
    <h6 class="mb-0">📊 Información del Producto Seleccionado</h6>
  </div>
  <div class="card-body">
    <div class="row" id="producto-info">
      <div class="col-md-12 text-center text-muted">
        Selecciona un producto para ver información detallada
      </div>
    </div>
  </div>
</div>

<!-- Script para información en tiempo real -->
<script>
document.addEventListener('DOMContentLoaded', function() {
    const selectCinta = document.getElementById('select-cinta');
    const stockInfo = document.getElementById('stock-info');
    const inputCantidad = document.getElementById('input-cantidad');
    const valorTotalInfo = document.getElementById('valor-total-info');
    const productoInfo = document.getElementById('producto-info');
    const selectUnidad = document.getElementById('select-unidad');
    
    function actualizarInfoProducto() {
        const optionSeleccionada = selectCinta.options[selectCinta.selectedIndex];
        
        if (optionSeleccionada.value) {
            const stock = parseFloat(optionSeleccionada.getAttribute('data-stock'));
            const unidad = optionSeleccionada.getAttribute('data-unidad');
            const precio = parseFloat(optionSeleccionada.getAttribute('data-precio'));
            const stockMinimo = parseFloat(optionSeleccionada.getAttribute('data-stock-minimo'));
            const texto = optionSeleccionada.text;
            
            // Actualizar información de stock
            let stockClass = 'text-success';
            if (stock < stockMinimo) stockClass = 'text-danger';
            else if (stock < (stockMinimo * 2)) stockClass = 'text-warning';
            
            stockInfo.innerHTML = `Stock disponible: <strong class="${stockClass}">${stock.toLocaleString()} ${unidad}</strong> | Mínimo: ${stockMinimo} ${unidad}`;
            
            // Actualizar select de unidad para que coincida con el producto
            selectUnidad.value = unidad;
            
            // Actualizar información del producto
            productoInfo.innerHTML = `
                <div class="col-md-3">
                    <strong>Producto:</strong><br>
                    <span class="text-primary">${texto.split(' (Stock:')[0]}</span>
                </div>
                <div class="col-md-3">
                    <strong>Stock Actual:</strong><br>
                    <span class="fw-bold ${stockClass}">${stock.toLocaleString()} ${unidad}</span>
                </div>
                <div class="col-md-3">
                    <strong>Precio Unitario:</strong><br>
                    <span class="text-success">$${precio.toLocaleString()} CLP</span>
                </div>
                <div class="col-md-3">
                    <strong>Estado Stock:</strong><br>
                    <span class="${stockClass}">${stock >= (stockMinimo * 2) ? '✅ Alto' : stock >= stockMinimo ? '🟡 Medio' : '🔴 Bajo'}</span>
                </div>
            `;
            
            // Calcular valor total si hay cantidad
            calcularValorTotal();
        } else {
            stockInfo.innerHTML = '';
            valorTotalInfo.innerHTML = '';
            productoInfo.innerHTML = '<div class="col-md-12 text-center text-muted">Selecciona un producto para ver información detallada</div>';
        }
    }
    
    function calcularValorTotal() {
        const optionSeleccionada = selectCinta.options[selectCinta.selectedIndex];
        const cantidad = parseFloat(inputCantidad.value) || 0;
        const precio = parseFloat(optionSeleccionada.getAttribute('data-precio')) || 0;
        
        if (cantidad > 0 && precio > 0) {
            const valorTotal = cantidad * precio;
            valorTotalInfo.innerHTML = `Valor total del ingreso: <strong>$${valorTotal.toLocaleString()} CLP</strong>`;
        } else {
            valorTotalInfo.innerHTML = 'Ingresa una cantidad para calcular el valor total';
        }
    }
    
    selectCinta.addEventListener('change', actualizarInfoProducto);
    inputCantidad.addEventListener('input', calcularValorTotal);
    
    // Ejecutar al cargar la página
    actualizarInfoProducto();
});
</script>

<?php include __DIR__ . '/../includes/footer.php'; ?>