<?php
require __DIR__ . '/../includes/auth_check.php';
require __DIR__ . '/../config/db.php';

$periodo = $_GET['periodo'] ?? 'mes';
$exportar = isset($_GET['exportar']) && $_GET['exportar'] === 'excel';
$categoriaFiltro = $_GET['categoria'] ?? 'all';
$fecha_inicio = $_GET['fecha_inicio'] ?? '';
$fecha_fin = $_GET['fecha_fin'] ?? '';

// Función para traducir unidad
function traducirUnidad($u) {
    switch ($u) {
        case 'm': return 'Metros';
        case 'kg': return 'Kilos';
        case 'u': return 'Unidades';
        default: return $u;
    }
}

// Función para formatear moneda
function formatearMoneda($valor) {
    return '$' . number_format($valor, 0, '', '.') . ' CLP';
}

// Función para exportar a Excel
function exportarExcel($reporte, $periodo, $titulo, $categoriaFiltro = 'all', $fecha_inicio = '', $fecha_fin = '') {
    header("Content-Type: application/vnd.ms-excel; charset=utf-8");
    header("Content-Disposition: attachment; filename=reporte_{$periodo}_" . date('Y-m-d') . ".xls");
    header("Pragma: no-cache");
    header("Expires: 0");

    echo "<html>";
    echo "<head>";
    echo "<meta charset='UTF-8'>";
    echo "<style>";
    echo "table { border-collapse: collapse; width: 100%; }";
    echo "th, td { border: 1px solid #000; padding: 8px; text-align: left; }";
    echo "th { background-color: #f2f2f2; font-weight: bold; }";
    echo ".number { text-align: right; }";
    echo ".text-center { text-align: center; }";
    echo ".total { background-color: #e9ecef; font-weight: bold; }";
    echo ".success { background-color: #d4edda; }";
    echo ".danger { background-color: #f8d7da; }";
    echo "</style>";
    echo "</head>";
    echo "<body>";
    
    echo "<h2>Reporte - $titulo</h2>";
    echo "<p><strong>Empresa:</strong> Jusocar</p>";
    echo "<p><strong>Generado:</strong> " . date('d/m/Y H:i:s') . "</p>";
    
    if ($fecha_inicio && $fecha_fin) {
        echo "<p><strong>Período:</strong> $fecha_inicio al $fecha_fin</p>";
    }
    
    if ($categoriaFiltro !== 'all') {
        global $pdo;
        $stmt = $pdo->prepare("SELECT nombre FROM categorias WHERE id = ?");
        $stmt->execute([$categoriaFiltro]);
        $categoria = $stmt->fetchColumn();
        echo "<p><strong>Categoría:</strong> " . ($categoria ?: 'Todas') . "</p>";
    }
    
    echo "<table>";
    
    if ($periodo === 'stock') {
        echo "<tr>";
        echo "<th>Categoría</th>";
        echo "<th>Código</th>";
        echo "<th>Descripción</th>";
        echo "<th>Unidad</th>";
        echo "<th>Precio Unitario</th>";
        echo "<th>Stock</th>";
        echo "<th>Stock Mínimo</th>";
        echo "<th>Valor Total</th>";
        echo "</tr>";
        
        $totalValor = 0;
        foreach ($reporte as $r) {
            $valorTotal = $r['stock'] * $r['precio_unitario'];
            $totalValor += $valorTotal;
            
            echo "<tr>";
            echo "<td>" . htmlspecialchars($r['categoria']) . "</td>";
            echo "<td>" . htmlspecialchars($r['codigo']) . "</td>";
            echo "<td>" . htmlspecialchars($r['descripcion']) . "</td>";
            echo "<td>" . traducirUnidad($r['unidad']) . "</td>";
            echo "<td class='number'>" . formatearMoneda($r['precio_unitario']) . "</td>";
            echo "<td class='number'>" . number_format($r['stock'], 0, '', '.') . "</td>";
            echo "<td class='number'>" . number_format($r['stock_minimo'], 0, '', '.') . "</td>";
            echo "<td class='number'>" . formatearMoneda($valorTotal) . "</td>";
            echo "</tr>";
        }
        
        echo "<tr class='total'>";
        echo "<td colspan='7' class='number'><strong>TOTAL GENERAL:</strong></td>";
        echo "<td class='number'><strong>" . formatearMoneda($totalValor) . "</strong></td>";
        echo "</tr>";
        
    } elseif ($periodo === 'personalizado') {
        echo "<tr>";
        echo "<th>Fecha</th>";
        echo "<th>Tipo</th>";
        echo "<th>Categoría</th>";
        echo "<th>Código</th>";
        echo "<th>Descripción</th>";
        echo "<th>Cantidad</th>";
        echo "<th>Unidad</th>";
        echo "<th>Precio Unitario</th>";
        echo "<th>Valor</th>";
        echo "<th>Documento</th>";
        echo "</tr>";
        
        $totalIngresos = 0;
        $totalSalidas = 0;
        foreach ($reporte as $r) {
            $valor = $r['cantidad'] * $r['precio_unitario'];
            $rowClass = $r['tipo'] === 'ingreso' ? 'success' : 'danger';
            
            if ($r['tipo'] === 'ingreso') {
                $totalIngresos += $valor;
            } else {
                $totalSalidas += $valor;
            }
            
            echo "<tr class='$rowClass'>";
            echo "<td>" . date('d/m/Y H:i', strtotime($r['fecha'])) . "</td>";
            echo "<td>" . ucfirst($r['tipo']) . "</td>";
            echo "<td>" . htmlspecialchars($r['categoria_nombre']) . "</td>";
            echo "<td>" . htmlspecialchars($r['codigo']) . "</td>";
            echo "<td>" . htmlspecialchars($r['descripcion']) . "</td>";
            echo "<td class='number'>" . number_format($r['cantidad'], 0, '', '.') . "</td>";
            echo "<td>" . traducirUnidad($r['unidad']) . "</td>";
            echo "<td class='number'>" . formatearMoneda($r['precio_unitario']) . "</td>";
            echo "<td class='number'>" . formatearMoneda($valor) . "</td>";
            echo "<td>" . $r['tipo_doc'] . ": " . $r['documento'] . "</td>";
            echo "</tr>";
        }
        
        echo "<tr class='total'>";
        echo "<td colspan='8' class='number'><strong>TOTAL INGRESOS:</strong></td>";
        echo "<td class='number'><strong>" . formatearMoneda($totalIngresos) . "</strong></td>";
        echo "<td></td>";
        echo "</tr>";
        
        echo "<tr class='total'>";
        echo "<td colspan='8' class='number'><strong>TOTAL SALIDAS:</strong></td>";
        echo "<td class='number'><strong>" . formatearMoneda($totalSalidas) . "</strong></td>";
        echo "<td></td>";
        echo "</tr>";
        
        echo "<tr class='total'>";
        echo "<td colspan='8' class='number'><strong>NETO:</strong></td>";
        echo "<td class='number'><strong>" . formatearMoneda($totalIngresos - $totalSalidas) . "</strong></td>";
        echo "<td></td>";
        echo "</tr>";
        
    } else {
        // Reportes semanal, mensual, anual
        echo "<tr>";
        echo "<th>Código</th>";
        echo "<th>Descripción</th>";
        echo "<th>Unidad</th>";
        echo "<th>Precio Unitario</th>";
        echo "<th>Ingresos</th>";
        echo "<th>Salidas</th>";
        echo "<th>Neto</th>";
        echo "<th>Valor Ingresos</th>";
        echo "<th>Valor Salidas</th>";
        echo "</tr>";
        
        $totalIngresos = 0;
        $totalSalidas = 0;
        $totalValorIngresos = 0;
        $totalValorSalidas = 0;
        
        foreach ($reporte as $r) {
            $totalIngresos += $r['ingresos'];
            $totalSalidas += $r['salidas'];
            $totalValorIngresos += $r['valor_ingresos'];
            $totalValorSalidas += $r['valor_salidas'];
            
            $netoClass = $r['movimiento_neto'] >= 0 ? 'success' : 'danger';
            
            echo "<tr>";
            echo "<td>" . htmlspecialchars($r['codigo']) . "</td>";
            echo "<td>" . htmlspecialchars($r['descripcion']) . "</td>";
            echo "<td>" . traducirUnidad($r['unidad']) . "</td>";
            echo "<td class='number'>" . formatearMoneda($r['precio_unitario']) . "</td>";
            echo "<td class='number'>" . number_format($r['ingresos'], 0, '', '.') . "</td>";
            echo "<td class='number'>" . number_format($r['salidas'], 0, '', '.') . "</td>";
            echo "<td class='number $netoClass'>" . number_format($r['movimiento_neto'], 0, '', '.') . "</td>";
            echo "<td class='number'>" . formatearMoneda($r['valor_ingresos']) . "</td>";
            echo "<td class='number'>" . formatearMoneda($r['valor_salidas']) . "</td>";
            echo "</tr>";
        }
        
        echo "<tr class='total'>";
        echo "<td colspan='4' class='number'><strong>TOTALES:</strong></td>";
        echo "<td class='number'><strong>" . number_format($totalIngresos, 0, '', '.') . "</strong></td>";
        echo "<td class='number'><strong>" . number_format($totalSalidas, 0, '', '.') . "</strong></td>";
        echo "<td class='number'><strong>" . number_format($totalIngresos - $totalSalidas, 0, '', '.') . "</strong></td>";
        echo "<td class='number'><strong>" . formatearMoneda($totalValorIngresos) . "</strong></td>";
        echo "<td class='number'><strong>" . formatearMoneda($totalValorSalidas) . "</strong></td>";
        echo "</tr>";
    }
    
    echo "</table>";
    echo "</body></html>";
    exit;
}

if ($periodo === 'stock') {
    // REPORTE DE STOCK COMPLETO
    $sqlBase = "
        SELECT c.codigo, c.descripcion, c.unidad, c.precio_unitario, c.stock, c.stock_minimo,
               cat.nombre AS categoria, cat.id AS categoria_id,
               (c.stock * c.precio_unitario) as valor_total
        FROM cintas c
        LEFT JOIN categorias cat ON c.categoria_id = cat.id
        WHERE c.activo = 1
    ";
    
    $condiciones = [];
    $params = [];

    if ($categoriaFiltro !== 'all') {
        $condiciones[] = "cat.id = :categoria";
        $params[':categoria'] = $categoriaFiltro;
    }
    
    if ($condiciones) {
        $sqlBase .= " AND " . implode(" AND ", $condiciones);
    }
    
    $sqlBase .= " ORDER BY cat.nombre, c.codigo";

    // EXPORTAR EXCEL
    if ($exportar) {
        $stmt = $pdo->prepare($sqlBase);
        $stmt->execute($params);
        $reporte = $stmt->fetchAll();
        exportarExcel($reporte, $periodo, "Stock Completo", $categoriaFiltro);
    }

    // PAGINACIÓN PARA VISTA EN PANTALLA
    $page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
    $limit = 25;
    $offset = ($page - 1) * $limit;

    $sqlCount = "SELECT COUNT(*) FROM cintas c LEFT JOIN categorias cat ON c.categoria_id = cat.id WHERE c.activo = 1";
    if ($categoriaFiltro !== 'all') {
        $sqlCount .= " AND cat.id = :categoria";
    }
    $stmtCount = $pdo->prepare($sqlCount);
    if ($categoriaFiltro !== 'all') {
        $stmtCount->bindValue(':categoria', $categoriaFiltro, PDO::PARAM_INT);
    }
    $stmtCount->execute();
    $totalProductos = $stmtCount->fetchColumn();
    $totalPaginas = ceil($totalProductos / $limit);

    $sqlPage = $sqlBase . " LIMIT :limit OFFSET :offset";
    $stmt = $pdo->prepare($sqlPage);
    foreach ($params as $k => $v) {
        $stmt->bindValue($k, $v, PDO::PARAM_INT);
    }
    $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
    $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
    $stmt->execute();
    $reporte = $stmt->fetchAll();

    $titulo = "Reporte de Todo el Stock";
    
} elseif ($periodo === 'personalizado') {
    // REPORTE PERSONALIZADO POR FECHAS
    $titulo = "Reporte Personalizado";
    
    if ($fecha_inicio && $fecha_fin) {
        $sql = "
            SELECT m.*, c.codigo, c.descripcion, c.unidad, c.precio_unitario,
                   cat.nombre as categoria_nombre
            FROM movimientos m
            JOIN cintas c ON m.cinta_id = c.id
            LEFT JOIN categorias cat ON c.categoria_id = cat.id
            WHERE DATE(m.fecha) BETWEEN :fecha_inicio AND :fecha_fin
            ORDER BY m.fecha DESC
        ";
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            ':fecha_inicio' => $fecha_inicio,
            ':fecha_fin' => $fecha_fin
        ]);
        $reporte = $stmt->fetchAll();
        
        // EXPORTAR EXCEL
        if ($exportar) {
            exportarExcel($reporte, $periodo, $titulo, 'all', $fecha_inicio, $fecha_fin);
        }
    } else {
        $reporte = [];
    }
    
} else {
    // REPORTES DE MOVIMIENTOS (SEMANA, MES, AÑO)
    switch ($periodo) {
        case 'semana':
            $condicion = "YEARWEEK(m.fecha, 1) = YEARWEEK(CURDATE(), 1)";
            $titulo = "Reporte Semanal (" . date('d/m/Y', strtotime('monday this week')) . " - " . date('d/m/Y', strtotime('sunday this week')) . ")";
            break;
        case 'anio':
            $condicion = "YEAR(m.fecha) = YEAR(CURDATE())";
            $titulo = "Reporte Anual " . date('Y');
            break;
        default:
            $condicion = "MONTH(m.fecha) = MONTH(CURDATE()) AND YEAR(m.fecha) = YEAR(CURDATE())";
            $titulo = "Reporte Mensual " . date('F Y');
            break;
    }

    $sql = "
        SELECT c.codigo, c.descripcion, c.unidad, c.precio_unitario,
               SUM(CASE WHEN m.tipo = 'ingreso' THEN m.cantidad ELSE 0 END) as ingresos,
               SUM(CASE WHEN m.tipo = 'salida' THEN m.cantidad ELSE 0 END) as salidas,
               SUM(CASE WHEN m.tipo = 'ingreso' THEN m.cantidad ELSE -m.cantidad END) AS movimiento_neto,
               SUM(CASE WHEN m.tipo = 'ingreso' THEN m.cantidad * c.precio_unitario ELSE 0 END) as valor_ingresos,
               SUM(CASE WHEN m.tipo = 'salida' THEN m.cantidad * c.precio_unitario ELSE 0 END) as valor_salidas
        FROM movimientos m
        JOIN cintas c ON m.cinta_id = c.id
        WHERE $condicion AND c.activo = 1
        GROUP BY c.id, c.codigo, c.descripcion, c.unidad, c.precio_unitario
        ORDER BY movimiento_neto DESC
    ";
    $stmt = $pdo->prepare($sql);
    $stmt->execute();
    $reporte = $stmt->fetchAll();
    
    // EXPORTAR EXCEL
    if ($exportar) {
        exportarExcel($reporte, $periodo, $titulo);
    }
}

include __DIR__ . '/../includes/header.php';
?>
<h3 class="mb-4">📊 <?= $titulo ?></h3>

<!-- Menú de Reportes -->
<div class="card mb-4">
    <div class="card-header bg-primary text-white">
        <h6 class="mb-0">Seleccionar Tipo de Reporte</h6>
    </div>
    <div class="card-body">
        <div class="row g-2">
            <div class="col-md-2">
                <a href="?periodo=semana" class="btn btn-outline-info w-100 <?= $periodo === 'semana' ? 'active' : '' ?>">
                    📅 Semanal
                </a>
            </div>
            <div class="col-md-2">
                <a href="?periodo=mes" class="btn btn-outline-warning w-100 <?= $periodo === 'mes' ? 'active' : '' ?>">
                    📊 Mensual
                </a>
            </div>
            <div class="col-md-2">
                <a href="?periodo=anio" class="btn btn-outline-success w-100 <?= $periodo === 'anio' ? 'active' : '' ?>">
                    📈 Anual
                </a>
            </div>
            <div class="col-md-2">
                <a href="?periodo=stock" class="btn btn-outline-secondary w-100 <?= $periodo === 'stock' ? 'active' : '' ?>">
                    📦 Stock
                </a>
            </div>
            <div class="col-md-2">
                <a href="?periodo=personalizado" class="btn btn-outline-primary w-100 <?= $periodo === 'personalizado' ? 'active' : '' ?>">
                    🗓️ Personalizado
                </a>
            </div>
            <div class="col-md-2">
                <a href="?periodo=<?= $periodo ?>&exportar=excel<?= $categoriaFiltro ? '&categoria='.$categoriaFiltro : '' ?><?= $fecha_inicio ? '&fecha_inicio='.$fecha_inicio.'&fecha_fin='.$fecha_fin : '' ?>" 
                   class="btn btn-success w-100">
                    ⬇️ Excel
                </a>
            </div>
        </div>
    </div>
</div>

<!-- Filtros Específicos -->
<?php if ($periodo === 'stock'): ?>
<!-- Filtro por categoría para reporte de stock -->
<form method="get" class="card p-3 mb-4 shadow-sm">
    <input type="hidden" name="periodo" value="stock">
    <div class="row g-3">
        <div class="col-md-6">
            <label class="form-label">Filtrar por Categoría</label>
            <select name="categoria" class="form-select">
                <option value="all">-- Todas las categorías --</option>
                <?php
                $cats = $pdo->query("SELECT * FROM categorias ORDER BY nombre")->fetchAll();
                foreach ($cats as $cat) {
                    $selected = ($categoriaFiltro == $cat['id']) ? 'selected' : '';
                    echo "<option value='{$cat['id']}' $selected>".htmlspecialchars($cat['nombre'])."</option>";
                }
                ?>
            </select>
        </div>
        <div class="col-md-4 d-flex align-items-end">
            <button type="submit" class="btn btn-primary w-100">🔍 Aplicar Filtro</button>
        </div>
        <div class="col-md-2 d-flex align-items-end">
            <a href="?periodo=stock" class="btn btn-outline-secondary w-100">🔄 Limpiar</a>
        </div>
    </div>
</form>

<?php elseif ($periodo === 'personalizado'): ?>
<!-- Filtro por fechas para reporte personalizado -->
<form method="get" class="card p-3 mb-4 shadow-sm">
    <input type="hidden" name="periodo" value="personalizado">
    <div class="row g-3">
        <div class="col-md-4">
            <label class="form-label">Fecha Inicio</label>
            <input type="date" name="fecha_inicio" value="<?= htmlspecialchars($fecha_inicio) ?>" class="form-control">
        </div>
        <div class="col-md-4">
            <label class="form-label">Fecha Fin</label>
            <input type="date" name="fecha_fin" value="<?= htmlspecialchars($fecha_fin) ?>" class="form-control">
        </div>
        <div class="col-md-2 d-flex align-items-end">
            <button type="submit" class="btn btn-primary w-100">🔍 Generar</button>
        </div>
        <div class="col-md-2 d-flex align-items-end">
            <a href="?periodo=personalizado" class="btn btn-outline-secondary w-100">🔄 Limpiar</a>
        </div>
    </div>
</form>
<?php endif; ?>

<!-- Estadísticas Rápidas -->
<?php if (in_array($periodo, ['semana', 'mes', 'anio']) && !empty($reporte)): ?>
<?php
$totalIngresos = array_sum(array_column($reporte, 'ingresos'));
$totalSalidas = array_sum(array_column($reporte, 'salidas'));
$valorTotalIngresos = array_sum(array_column($reporte, 'valor_ingresos'));
$valorTotalSalidas = array_sum(array_column($reporte, 'valor_salidas'));
?>
<div class="row mb-4">
    <div class="col-md-3">
        <div class="card bg-success text-white">
            <div class="card-body text-center py-3">
                <h5><?= number_format($totalIngresos, 0, '', '.') ?></h5>
                <small>Total Ingresos</small>
                <br>
                <small><?= formatearMoneda($valorTotalIngresos) ?></small>
            </div>
        </div>
    </div>
    <div class="col-md-3">
        <div class="card bg-danger text-white">
            <div class="card-body text-center py-3">
                <h5><?= number_format($totalSalidas, 0, '', '.') ?></h5>
                <small>Total Salidas</small>
                <br>
                <small><?= formatearMoneda($valorTotalSalidas) ?></small>
            </div>
        </div>
    </div>
    <div class="col-md-3">
        <div class="card bg-info text-white">
            <div class="card-body text-center py-3">
                <h5><?= number_format($totalIngresos - $totalSalidas, 0, '', '.') ?></h5>
                <small>Movimiento Neto</small>
            </div>
        </div>
    </div>
    <div class="col-md-3">
        <div class="card bg-warning text-white">
            <div class="card-body text-center py-3">
                <h5><?= count($reporte) ?></h5>
                <small>Productos con Movimiento</small>
            </div>
        </div>
    </div>
</div>
<?php endif; ?>

<!-- Tabla de Reporte -->
<div class="table-responsive">
    <table class="table table-striped table-hover shadow-sm">
        <thead class="table-primary">
            <?php if ($periodo === 'stock'): ?>
                <tr>
                    <th>Categoría</th>
                    <th>Código</th>
                    <th>Descripción</th>
                    <th>Unidad</th>
                    <th>Precio Unitario</th>
                    <th>Stock</th>
                    <th>Stock Mínimo</th>
                    <th>Valor Total</th>
                </tr>
            <?php elseif ($periodo === 'personalizado'): ?>
                <tr>
                    <th>Fecha</th>
                    <th>Tipo</th>
                    <th>Categoría</th>
                    <th>Código</th>
                    <th>Descripción</th>
                    <th>Cantidad</th>
                    <th>Unidad</th>
                    <th>Precio Unitario</th>
                    <th>Valor</th>
                    <th>Documento</th>
                </tr>
            <?php else: ?>
                <tr>
                    <th>Código</th>
                    <th>Descripción</th>
                    <th>Unidad</th>
                    <th>Precio Unitario</th>
                    <th>Ingresos</th>
                    <th>Salidas</th>
                    <th>Neto</th>
                    <th>Valor Ingresos</th>
                    <th>Valor Salidas</th>
                </tr>
            <?php endif; ?>
        </thead>
        <tbody>
            <?php if (empty($reporte)): ?>
                <tr>
                    <td colspan="<?= $periodo === 'stock' ? 8 : ($periodo === 'personalizado' ? 10 : 9) ?>" class="text-center py-4">
                        <div class="text-muted">
                            <h5>📭 No se encontraron datos</h5>
                            <p><?= $periodo === 'personalizado' ? 'Selecciona un rango de fechas válido' : 'No hay movimientos en el período seleccionado' ?></p>
                        </div>
                    </td>
                </tr>
            <?php else: ?>
                <?php 
                $totalGeneral = 0;
                foreach ($reporte as $r): 
                    if ($periodo === 'stock') {
                        $valorTotal = $r['stock'] * $r['precio_unitario'];
                        $totalGeneral += $valorTotal;
                    }
                ?>
                    <?php if ($periodo === 'stock'): ?>
                        <tr>
                            <td>
                                <span class="badge bg-secondary"><?= htmlspecialchars($r['categoria']) ?></span>
                            </td>
                            <td><strong><?= htmlspecialchars($r['codigo']) ?></strong></td>
                            <td><?= htmlspecialchars($r['descripcion']) ?></td>
                            <td><?= traducirUnidad($r['unidad']) ?></td>
                            <td class="text-end"><?= formatearMoneda($r['precio_unitario']) ?></td>
                            <td class="text-end <?= $r['stock'] < $r['stock_minimo'] ? 'text-danger fw-bold' : '' ?>">
                                <?= number_format($r['stock'], 0, '', '.') ?>
                            </td>
                            <td class="text-end"><?= number_format($r['stock_minimo'], 0, '', '.') ?></td>
                            <td class="text-end fw-bold"><?= formatearMoneda($valorTotal) ?></td>
                        </tr>
                    <?php elseif ($periodo === 'personalizado'): ?>
                        <tr class="<?= $r['tipo'] === 'ingreso' ? 'bg-light-green' : 'bg-light-red' ?>">
                            <td><?= date('d/m/Y H:i', strtotime($r['fecha'])) ?></td>
                            <td>
                                <span class="badge bg-<?= $r['tipo'] === 'ingreso' ? 'success' : 'danger' ?>">
                                    <?= ucfirst($r['tipo']) ?>
                                </span>
                            </td>
                            <td><?= htmlspecialchars($r['categoria_nombre']) ?></td>
                            <td><strong><?= htmlspecialchars($r['codigo']) ?></strong></td>
                            <td><?= htmlspecialchars($r['descripcion']) ?></td>
                            <td class="text-end"><?= number_format($r['cantidad'], 0, '', '.') ?></td>
                            <td><?= traducirUnidad($r['unidad']) ?></td>
                            <td class="text-end"><?= formatearMoneda($r['precio_unitario']) ?></td>
                            <td class="text-end fw-bold"><?= formatearMoneda($r['cantidad'] * $r['precio_unitario']) ?></td>
                            <td>
                                <small><?= $r['tipo_doc'] ?>: <?= $r['documento'] ?></small>
                            </td>
                        </tr>
                    <?php else: ?>
                        <tr>
                            <td><strong><?= htmlspecialchars($r['codigo']) ?></strong></td>
                            <td><?= htmlspecialchars($r['descripcion']) ?></td>
                            <td><?= traducirUnidad($r['unidad']) ?></td>
                            <td class="text-end"><?= formatearMoneda($r['precio_unitario']) ?></td>
                            <td class="text-end text-success"><?= number_format($r['ingresos'], 0, '', '.') ?></td>
                            <td class="text-end text-danger"><?= number_format($r['salidas'], 0, '', '.') ?></td>
                            <td class="text-end fw-bold <?= $r['movimiento_neto'] >= 0 ? 'text-success' : 'text-danger' ?>">
                                <?= number_format($r['movimiento_neto'], 0, '', '.') ?>
                            </td>
                            <td class="text-end text-success"><?= formatearMoneda($r['valor_ingresos']) ?></td>
                            <td class="text-end text-danger"><?= formatearMoneda($r['valor_salidas']) ?></td>
                        </tr>
                    <?php endif; ?>
                <?php endforeach; ?>
                
                <!-- Total General para reporte de stock -->
                <?php if ($periodo === 'stock'): ?>
                    <tr class="table-primary fw-bold">
                        <td colspan="7" class="text-end">TOTAL GENERAL:</td>
                        <td class="text-end"><?= formatearMoneda($totalGeneral) ?></td>
                    </tr>
                <?php endif; ?>
            <?php endif; ?>
        </tbody>
    </table>
</div>

<!-- Paginación para reporte de stock -->
<?php if ($periodo === 'stock' && $totalPaginas > 1): ?>
<nav aria-label="Paginación">
    <ul class="pagination justify-content-center">
        <li class="page-item <?= $page <= 1 ? 'disabled' : '' ?>">
            <a class="page-link" href="?<?= http_build_query(array_merge($_GET, ['page' => $page - 1])) ?>">
                ‹ Anterior
            </a>
        </li>
        
        <?php for ($i = 1; $i <= $totalPaginas; $i++): ?>
            <?php if ($i == 1 || $i == $totalPaginas || ($i >= $page - 2 && $i <= $page + 2)): ?>
                <li class="page-item <?= ($i == $page) ? 'active' : '' ?>">
                    <a class="page-link" href="?<?= http_build_query(array_merge($_GET, ['page' => $i])) ?>">
                        <?= $i ?>
                    </a>
                </li>
            <?php elseif ($i == $page - 3 || $i == $page + 3): ?>
                <li class="page-item disabled">
                    <span class="page-link">...</span>
                </li>
            <?php endif; ?>
        <?php endfor; ?>
        
        <li class="page-item <?= $page >= $totalPaginas ? 'disabled' : '' ?>">
            <a class="page-link" href="?<?= http_build_query(array_merge($_GET, ['page' => $page + 1])) ?>">
                Siguiente ›
            </a>
        </li>
    </ul>
</nav>

<div class="text-center text-muted mt-2">
    Página <?= $page ?> de <?= $totalPaginas ?> 
    • <?= number_format($totalProductos, 0, '', '.') ?> productos en total
</div>
<?php endif; ?>

<?php include __DIR__ . '/../includes/footer.php'; ?>